#ifndef __ATLSCINTILLA_H__
#define __ATLSCINTILLA_H__

#pragma once

#ifndef PLATFORM_H
#include <Platform.h>
#endif

#ifndef SCINTILLA_H
#include <Scintilla.h>
#endif

#ifndef SCILEXER_H
#include <SciLexer.h>
#endif

namespace WTL {

class CScintillaLib {
public:
	CScintillaLib() {
		if(++m_nRef==1)
			m_hLib = ::LoadLibrary(TEXT("SciLexer.dll"));
		ATLASSERT(m_hLib && TEXT("Failed to load SciLexer.dll"));
	}
	~CScintillaLib() {
		if(--m_nRef==0) {
			::FreeLibrary(m_hLib);
			m_hLib = NULL;
		}
	}
protected:
	static HMODULE	m_hLib;
	static long		m_nRef;
};

__declspec(selectany) HMODULE	CScintillaLib::m_hLib;
__declspec(selectany) long		CScintillaLib::m_nRef;

template<class TBase>
class CScintillaT : public TBase, public CScintillaLib {
public:
// Constructors
	CScintillaT(HWND hWnd = NULL) : TBase(hWnd) { }

	CScintillaT< TBase >& operator=(HWND hWnd) {
		m_hWnd = hWnd;
		return *this;
	}

	HWND Create(HWND hWndParent, _U_RECT rect = NULL, LPCTSTR szWindowName = NULL,
			DWORD dwStyle = 0, DWORD dwExStyle = 0,
			_U_MENUorID MenuOrID = 0U, LPVOID lpCreateParam = NULL)
	{
		return CWindow::Create(GetWndClassName(), hWndParent, rect.m_lpRect, szWindowName, dwStyle, dwExStyle, MenuOrID.m_hMenu, lpCreateParam);
	}

// Attributes
	static LPCTSTR GetWndClassName()
	{
		return _T("Scintilla");
	}

	//++Autogenerated
	/// Add text to the document at current position.
	void AddText(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ADDTEXT,(WPARAM)length,(LPARAM)text);
	}

	/// Add array of cells to document.
	void AddStyledText(int length,LPBYTE c) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ADDSTYLEDTEXT,(WPARAM)length,(LPARAM)c);
	}

	/// Insert string at a position.
	void InsertText(int pos,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_INSERTTEXT,(WPARAM)pos,(LPARAM)text);
	}

	/// Delete all text in the document.
	void ClearAll() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEARALL,(WPARAM)0,(LPARAM)0);
	}

	/// Set all style bytes to 0, remove all folding information.
	void ClearDocumentStyle() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEARDOCUMENTSTYLE,(WPARAM)0,(LPARAM)0);
	}

	/// Returns the number of characters in the document.
	int GetLength() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLENGTH,(WPARAM)0,(LPARAM)0);
	}

	/// Returns the character byte at the position.
	int GetCharAt(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCHARAT,(WPARAM)pos,(LPARAM)0);
	}

	/// Returns the position of the caret.
	int GetCurrentPos() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCURRENTPOS,(WPARAM)0,(LPARAM)0);
	}

	/// Returns the position of the opposite end of the selection to the caret.
	int GetAnchor() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETANCHOR,(WPARAM)0,(LPARAM)0);
	}

	/// Returns the style byte at the position.
	int GetStyleAt(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSTYLEAT,(WPARAM)pos,(LPARAM)0);
	}

	/// Redoes the next action on the undo history.
	void Redo() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_REDO,(WPARAM)0,(LPARAM)0);
	}

	/// Choose between collecting actions into the undo
	/// history and discarding them.
	void SetUndoCollection(bool collectUndo) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETUNDOCOLLECTION,(WPARAM)collectUndo,(LPARAM)0);
	}

	/// Select all the text in the document.
	void SelectAll() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SELECTALL,(WPARAM)0,(LPARAM)0);
	}

	/// Remember the current position in the undo history as the position
	/// at which the document was saved.
	void SetSavePoint() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSAVEPOINT,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve a buffer of cells.
	/// Returns the number of bytes in the buffer not including terminating NULs.
	int GetStyledText(TextRange* tr) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSTYLEDTEXT,(WPARAM)0,(LPARAM)tr);
	}

	/// Are there any redoable actions in the undo history?
	bool CanRedo() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_CANREDO,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Retrieve the line number at which a particular marker is located.
	int MarkerLineFromHandle(int handle) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_MARKERLINEFROMHANDLE,(WPARAM)handle,(LPARAM)0);
	}

	/// Delete a marker.
	void MarkerDeleteHandle(int handle) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERDELETEHANDLE,(WPARAM)handle,(LPARAM)0);
	}

	/// Is undo history being collected?
	bool GetUndoCollection() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETUNDOCOLLECTION,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Are white space characters currently visible?
	/// Returns one of SCWS_* constants.
	int GetViewWS() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETVIEWWS,(WPARAM)0,(LPARAM)0);
	}

	/// Make white space characters invisible, always visible or visible outside indentation.
	void SetViewWS(int viewWS) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETVIEWWS,(WPARAM)viewWS,(LPARAM)0);
	}

	/// Find the position from a point within the window.
	int PositionFromPoint(int x,int y) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POSITIONFROMPOINT,(WPARAM)x,(LPARAM)y);
	}

	/// Find the position from a point within the window but return
	/// INVALID_POSITION if not close to text.
	int PositionFromPointClose(int x,int y) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POSITIONFROMPOINTCLOSE,(WPARAM)x,(LPARAM)y);
	}

	/// Set caret to start of a line and ensure it is visible.
	void GotoLine(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_GOTOLINE,(WPARAM)line,(LPARAM)0);
	}

	/// Set caret to a position and ensure it is visible.
	void GotoPos(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_GOTOPOS,(WPARAM)pos,(LPARAM)0);
	}

	/// Set the selection anchor to a position. The anchor is the opposite
	/// end of the selection from the caret.
	void SetAnchor(int posAnchor) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETANCHOR,(WPARAM)posAnchor,(LPARAM)0);
	}

	/// Retrieve the text of the line containing the caret.
	/// Returns the index of the caret on the line.
	int GetCurLine(int length,char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCURLINE,(WPARAM)length,(LPARAM)text);
	}

	/// Retrieve the position of the last correctly styled character.
	int GetEndStyled() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETENDSTYLED,(WPARAM)0,(LPARAM)0);
	}

	/// Convert all line endings in the document to one mode.
	void ConvertEOLs(int eolMode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CONVERTEOLS,(WPARAM)eolMode,(LPARAM)0);
	}

	/// Retrieve the current end of line mode - one of CRLF, CR, or LF.
	int GetEOLMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETEOLMODE,(WPARAM)0,(LPARAM)0);
	}

	/// Set the current end of line mode.
	void SetEOLMode(int eolMode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETEOLMODE,(WPARAM)eolMode,(LPARAM)0);
	}

	/// Set the current styling position to pos and the styling mask to mask.
	/// The styling mask can be used to protect some bits in each styling byte from modification.
	void StartStyling(int pos,int mask) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STARTSTYLING,(WPARAM)pos,(LPARAM)mask);
	}

	/// Change style from current styling position for length characters to a style
	/// and move the current styling position to after this newly styled segment.
	void SetStyling(int length,int style) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSTYLING,(WPARAM)length,(LPARAM)style);
	}

	/// Is drawing done first into a buffer or direct to the screen?
	bool GetBufferedDraw() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETBUFFEREDDRAW,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// If drawing is buffered then each line of text is drawn into a bitmap buffer
	/// before drawing it to the screen to avoid flicker.
	void SetBufferedDraw(bool buffered) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETBUFFEREDDRAW,(WPARAM)buffered,(LPARAM)0);
	}

	/// Change the visible size of a tab to be a multiple of the width of a space character.
	void SetTabWidth(int tabWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTABWIDTH,(WPARAM)tabWidth,(LPARAM)0);
	}

	/// Retrieve the visible size of a tab.
	int GetTabWidth() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTABWIDTH,(WPARAM)0,(LPARAM)0);
	}

	/// Set the code page used to interpret the bytes of the document as characters.
	/// The SC_CP_UTF8 value can be used to enter Unicode mode.
	void SetCodePage(int codePage) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCODEPAGE,(WPARAM)codePage,(LPARAM)0);
	}

	/// In palette mode, Scintilla uses the environment's palette calls to display
	/// more colours. This may lead to ugly displays.
	void SetUsePalette(bool usePalette) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETUSEPALETTE,(WPARAM)usePalette,(LPARAM)0);
	}

	/// Set the symbol used for a particular marker number.
	void MarkerDefine(int markerNumber,int markerSymbol) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERDEFINE,(WPARAM)markerNumber,(LPARAM)markerSymbol);
	}

	/// Set the foreground colour used for a particular marker number.
	void MarkerSetFore(int markerNumber,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERSETFORE,(WPARAM)markerNumber,(LPARAM)fore);
	}

	/// Set the background colour used for a particular marker number.
	void MarkerSetBack(int markerNumber,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERSETBACK,(WPARAM)markerNumber,(LPARAM)back);
	}

	/// Add a marker to a line, returning an ID which can be used to find or delete the marker.
	int MarkerAdd(int line,int markerNumber) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_MARKERADD,(WPARAM)line,(LPARAM)markerNumber);
	}

	/// Delete a marker from a line.
	void MarkerDelete(int line,int markerNumber) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERDELETE,(WPARAM)line,(LPARAM)markerNumber);
	}

	/// Delete all markers with a particular number from all lines.
	void MarkerDeleteAll(int markerNumber) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERDELETEALL,(WPARAM)markerNumber,(LPARAM)0);
	}

	/// Get a bit mask of all the markers set on a line.
	int MarkerGet(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_MARKERGET,(WPARAM)line,(LPARAM)0);
	}

	/// Find the next line after lineStart that includes a marker in mask.
	int MarkerNext(int lineStart,int markerMask) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_MARKERNEXT,(WPARAM)lineStart,(LPARAM)markerMask);
	}

	/// Find the previous line before lineStart that includes a marker in mask.
	int MarkerPrevious(int lineStart,int markerMask) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_MARKERPREVIOUS,(WPARAM)lineStart,(LPARAM)markerMask);
	}

	/// Define a marker from a pixmap.
	void MarkerDefinePixmap(int markerNumber,const char* pixmap) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MARKERDEFINEPIXMAP,(WPARAM)markerNumber,(LPARAM)pixmap);
	}

	/// Set a margin to be either numeric or symbolic.
	void SetMarginTypeN(int margin,int marginType) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINTYPEN,(WPARAM)margin,(LPARAM)marginType);
	}

	/// Retrieve the type of a margin.
	int GetMarginTypeN(int margin) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMARGINTYPEN,(WPARAM)margin,(LPARAM)0);
	}

	/// Set the width of a margin to a width expressed in pixels.
	void SetMarginWidthN(int margin,int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINWIDTHN,(WPARAM)margin,(LPARAM)pixelWidth);
	}

	/// Retrieve the width of a margin in pixels.
	int GetMarginWidthN(int margin) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMARGINWIDTHN,(WPARAM)margin,(LPARAM)0);
	}

	/// Set a mask that determines which markers are displayed in a margin.
	void SetMarginMaskN(int margin,int mask) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINMASKN,(WPARAM)margin,(LPARAM)mask);
	}

	/// Retrieve the marker mask of a margin.
	int GetMarginMaskN(int margin) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMARGINMASKN,(WPARAM)margin,(LPARAM)0);
	}

	/// Make a margin sensitive or insensitive to mouse clicks.
	void SetMarginSensitiveN(int margin,bool sensitive) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINSENSITIVEN,(WPARAM)margin,(LPARAM)sensitive);
	}

	/// Retrieve the mouse click sensitivity of a margin.
	bool GetMarginSensitiveN(int margin) {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETMARGINSENSITIVEN,(WPARAM)margin,(LPARAM)0)!=0;
	}

	/// Clear all the styles and make equivalent to the global default style.
	void StyleClearAll() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLECLEARALL,(WPARAM)0,(LPARAM)0);
	}

	/// Set the foreground colour of a style.
	void StyleSetFore(int style,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETFORE,(WPARAM)style,(LPARAM)fore);
	}

	/// Set the background colour of a style.
	void StyleSetBack(int style,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETBACK,(WPARAM)style,(LPARAM)back);
	}

	/// Set a style to be bold or not.
	void StyleSetBold(int style,bool bold) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETBOLD,(WPARAM)style,(LPARAM)bold);
	}

	/// Set a style to be italic or not.
	void StyleSetItalic(int style,bool italic) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETITALIC,(WPARAM)style,(LPARAM)italic);
	}

	/// Set the size of characters of a style.
	void StyleSetSize(int style,int sizePoints) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETSIZE,(WPARAM)style,(LPARAM)sizePoints);
	}

	/// Set the font of a style.
	void StyleSetFont(int style,const char* fontName) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETFONT,(WPARAM)style,(LPARAM)fontName);
	}

	/// Set a style to have its end of line filled or not.
	void StyleSetEOLFilled(int style,bool filled) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETEOLFILLED,(WPARAM)style,(LPARAM)filled);
	}

	/// Reset the default style to its state at startup
	void StyleResetDefault() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLERESETDEFAULT,(WPARAM)0,(LPARAM)0);
	}

	/// Set a style to be underlined or not.
	void StyleSetUnderline(int style,bool underline) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETUNDERLINE,(WPARAM)style,(LPARAM)underline);
	}

	/// Set a style to be mixed case, or to force upper or lower case.
	void StyleSetCase(int style,int caseForce) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETCASE,(WPARAM)style,(LPARAM)caseForce);
	}

	/// Set the character set of the font in a style.
	void StyleSetCharacterSet(int style,int characterSet) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETCHARACTERSET,(WPARAM)style,(LPARAM)characterSet);
	}

	/// Set a style to be a hotspot or not.
	void StyleSetHotSpot(int style,bool hotspot) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETHOTSPOT,(WPARAM)style,(LPARAM)hotspot);
	}

	/// Set the foreground colour of the selection and whether to use this setting.
	void SetSelFore(bool useSetting,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSELFORE,(WPARAM)useSetting,(LPARAM)fore);
	}

	/// Set the background colour of the selection and whether to use this setting.
	void SetSelBack(bool useSetting,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSELBACK,(WPARAM)useSetting,(LPARAM)back);
	}

	/// Set the foreground colour of the caret.
	void SetCaretFore(COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCARETFORE,(WPARAM)fore,(LPARAM)0);
	}

	/// When key+modifier combination km is pressed perform msg.
	void AssignCmdKey(DWORD km,int msg) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ASSIGNCMDKEY,(WPARAM)km,(LPARAM)msg);
	}

	/// When key+modifier combination km is pressed do nothing.
	void ClearCmdKey(DWORD km) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEARCMDKEY,(WPARAM)km,(LPARAM)0);
	}

	/// Drop all key mappings.
	void ClearAllCmdKeys() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEARALLCMDKEYS,(WPARAM)0,(LPARAM)0);
	}

	/// Set the styles for a segment of the document.
	void SetStylingEx(int length,const char* styles) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSTYLINGEX,(WPARAM)length,(LPARAM)styles);
	}

	/// Set a style to be visible or not.
	void StyleSetVisible(int style,bool visible) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETVISIBLE,(WPARAM)style,(LPARAM)visible);
	}

	/// Get the time in milliseconds that the caret is on and off.
	int GetCaretPeriod() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCARETPERIOD,(WPARAM)0,(LPARAM)0);
	}

	/// Get the time in milliseconds that the caret is on and off. 0 = steady on.
	void SetCaretPeriod(int periodMilliseconds) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCARETPERIOD,(WPARAM)periodMilliseconds,(LPARAM)0);
	}

	/// Set the set of characters making up words for when moving or selecting by word.
	/// First sets deaults like SetCharsDefault.
	void SetWordChars(const char* characters) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWORDCHARS,(WPARAM)0,(LPARAM)characters);
	}

	/// Start a sequence of actions that is undone and redone as a unit.
	/// May be nested.
	void BeginUndoAction() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_BEGINUNDOACTION,(WPARAM)0,(LPARAM)0);
	}

	/// End a sequence of actions that is undone and redone as a unit.
	void EndUndoAction() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ENDUNDOACTION,(WPARAM)0,(LPARAM)0);
	}

	/// Set an indicator to plain, squiggle or TT.
	void IndicSetStyle(int indic,int style) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_INDICSETSTYLE,(WPARAM)indic,(LPARAM)style);
	}

	/// Retrieve the style of an indicator.
	int IndicGetStyle(int indic) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_INDICGETSTYLE,(WPARAM)indic,(LPARAM)0);
	}

	/// Set the foreground colour of an indicator.
	void IndicSetFore(int indic,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_INDICSETFORE,(WPARAM)indic,(LPARAM)fore);
	}

	/// Retrieve the foreground colour of an indicator.
	COLORREF IndicGetFore(int indic) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (COLORREF)::SendMessage(m_hWnd,SCI_INDICGETFORE,(WPARAM)indic,(LPARAM)0);
	}

	/// Set the foreground colour of all whitespace and whether to use this setting.
	void SetWhitespaceFore(bool useSetting,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWHITESPACEFORE,(WPARAM)useSetting,(LPARAM)fore);
	}

	/// Set the background colour of all whitespace and whether to use this setting.
	void SetWhitespaceBack(bool useSetting,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWHITESPACEBACK,(WPARAM)useSetting,(LPARAM)back);
	}

	/// Divide each styling byte into lexical class bits (default: 5) and indicator
	/// bits (default: 3). If a lexer requires more than 32 lexical states, then this
	/// is used to expand the possible states.
	void SetStyleBits(int bits) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSTYLEBITS,(WPARAM)bits,(LPARAM)0);
	}

	/// Retrieve number of bits in style bytes used to hold the lexical state.
	int GetStyleBits() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSTYLEBITS,(WPARAM)0,(LPARAM)0);
	}

	/// Used to hold extra styling information for each line.
	void SetLineState(int line,int state) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETLINESTATE,(WPARAM)line,(LPARAM)state);
	}

	/// Retrieve the extra styling information for a line.
	int GetLineState(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINESTATE,(WPARAM)line,(LPARAM)0);
	}

	/// Retrieve the last line number that has line state.
	int GetMaxLineState() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMAXLINESTATE,(WPARAM)0,(LPARAM)0);
	}

	/// Is the background of the line containing the caret in a different colour?
	bool GetCaretLineVisible() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETCARETLINEVISIBLE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Display the background of the line containing the caret in a different colour.
	void SetCaretLineVisible(bool show) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCARETLINEVISIBLE,(WPARAM)show,(LPARAM)0);
	}

	/// Get the colour of the background of the line containing the caret.
	COLORREF GetCaretLineBack() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (COLORREF)::SendMessage(m_hWnd,SCI_GETCARETLINEBACK,(WPARAM)0,(LPARAM)0);
	}

	/// Set the colour of the background of the line containing the caret.
	void SetCaretLineBack(COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCARETLINEBACK,(WPARAM)back,(LPARAM)0);
	}

	/// Set a style to be changeable or not (read only).
	/// Experimental feature, currently buggy.
	void StyleSetChangeable(int style,bool changeable) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STYLESETCHANGEABLE,(WPARAM)style,(LPARAM)changeable);
	}

	/// Display a auto-completion list.
	/// The lenEntered parameter indicates how many characters before
	/// the caret should be used to provide context.
	void AutoCShow(int lenEntered,const char* itemList) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSHOW,(WPARAM)lenEntered,(LPARAM)itemList);
	}

	/// Remove the auto-completion list from the screen.
	void AutoCCancel() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCCANCEL,(WPARAM)0,(LPARAM)0);
	}

	/// Is there an auto-completion list visible?
	bool AutoCActive() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCACTIVE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Retrieve the position of the caret when the auto-completion list was displayed.
	int AutoCPosStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_AUTOCPOSSTART,(WPARAM)0,(LPARAM)0);
	}

	/// User has selected an item so remove the list and insert the selection.
	void AutoCComplete() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCCOMPLETE,(WPARAM)0,(LPARAM)0);
	}

	/// Define a set of character that when typed cancel the auto-completion list.
	void AutoCStops(const char* characterSet) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSTOPS,(WPARAM)0,(LPARAM)characterSet);
	}

	/// Change the separator character in the string setting up an auto-completion list.
	/// Default is space but can be changed if items contain space.
	void AutoCSetSeparator(int separatorCharacter) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETSEPARATOR,(WPARAM)separatorCharacter,(LPARAM)0);
	}

	/// Retrieve the auto-completion list separator character.
	int AutoCGetSeparator() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_AUTOCGETSEPARATOR,(WPARAM)0,(LPARAM)0);
	}

	/// Select the item in the auto-completion list that starts with a string.
	void AutoCSelect(const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSELECT,(WPARAM)0,(LPARAM)text);
	}

	/// Should the auto-completion list be cancelled if the user backspaces to a
	/// position before where the box was created.
	void AutoCSetCancelAtStart(bool cancel) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETCANCELATSTART,(WPARAM)cancel,(LPARAM)0);
	}

	/// Retrieve whether auto-completion cancelled by backspacing before start.
	bool AutoCGetCancelAtStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCGETCANCELATSTART,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Define a set of characters that when typed will cause the autocompletion to
	/// choose the selected item.
	void AutoCSetFillUps(const char* characterSet) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETFILLUPS,(WPARAM)0,(LPARAM)characterSet);
	}

	/// Should a single item auto-completion list automatically choose the item.
	void AutoCSetChooseSingle(bool chooseSingle) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETCHOOSESINGLE,(WPARAM)chooseSingle,(LPARAM)0);
	}

	/// Retrieve whether a single item auto-completion list automatically choose the item.
	bool AutoCGetChooseSingle() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCGETCHOOSESINGLE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Set whether case is significant when performing auto-completion searches.
	void AutoCSetIgnoreCase(bool ignoreCase) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETIGNORECASE,(WPARAM)ignoreCase,(LPARAM)0);
	}

	/// Retrieve state of ignore case flag.
	bool AutoCGetIgnoreCase() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCGETIGNORECASE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Display a list of strings and send notification when user chooses one.
	void UserListShow(int listType,const char* itemList) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_USERLISTSHOW,(WPARAM)listType,(LPARAM)itemList);
	}

	/// Set whether or not autocompletion is hidden automatically when nothing matches.
	void AutoCSetAutoHide(bool autoHide) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETAUTOHIDE,(WPARAM)autoHide,(LPARAM)0);
	}

	/// Retrieve whether or not autocompletion is hidden automatically when nothing matches.
	bool AutoCGetAutoHide() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCGETAUTOHIDE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Set whether or not autocompletion deletes any word characters
	/// after the inserted text upon completion.
	void AutoCSetDropRestOfWord(bool dropRestOfWord) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETDROPRESTOFWORD,(WPARAM)dropRestOfWord,(LPARAM)0);
	}

	/// Retrieve whether or not autocompletion deletes any word characters
	/// after the inserted text upon completion.
	bool AutoCGetDropRestOfWord() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_AUTOCGETDROPRESTOFWORD,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Register an XPM image for use in autocompletion lists.
	void RegisterImage(int type,const char* xpmData) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_REGISTERIMAGE,(WPARAM)type,(LPARAM)xpmData);
	}

	/// Clear all the registered XPM images.
	void ClearRegisteredImages() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEARREGISTEREDIMAGES,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve the auto-completion list type-separator character.
	int AutoCGetTypeSeparator() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_AUTOCGETTYPESEPARATOR,(WPARAM)0,(LPARAM)0);
	}

	/// Change the type-separator character in the string setting up an auto-completion list.
	/// Default is '?' but can be changed if items contain '?'.
	void AutoCSetTypeSeparator(int separatorCharacter) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_AUTOCSETTYPESEPARATOR,(WPARAM)separatorCharacter,(LPARAM)0);
	}

	/// Set the number of spaces used for one level of indentation.
	void SetIndent(int indentSize) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETINDENT,(WPARAM)indentSize,(LPARAM)0);
	}

	/// Retrieve indentation size.
	int GetIndent() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETINDENT,(WPARAM)0,(LPARAM)0);
	}

	/// Indentation will only use space characters if useTabs is false, otherwise
	/// it will use a combination of tabs and spaces.
	void SetUseTabs(bool useTabs) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETUSETABS,(WPARAM)useTabs,(LPARAM)0);
	}

	/// Retrieve whether tabs will be used in indentation.
	bool GetUseTabs() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETUSETABS,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Change the indentation of a line to a number of columns.
	void SetLineIndentation(int line,int indentSize) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETLINEINDENTATION,(WPARAM)line,(LPARAM)indentSize);
	}

	/// Retrieve the number of columns that a line is indented.
	int GetLineIndentation(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINEINDENTATION,(WPARAM)line,(LPARAM)0);
	}

	/// Retrieve the position before the first non indentation character on a line.
	int GetLineIndentPosition(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINEINDENTPOSITION,(WPARAM)line,(LPARAM)0);
	}

	/// Retrieve the column number of a position, taking tab width into account.
	int GetColumn(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCOLUMN,(WPARAM)pos,(LPARAM)0);
	}

	/// Show or hide the horizontal scroll bar.
	void SetHScrollBar(bool show) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHSCROLLBAR,(WPARAM)show,(LPARAM)0);
	}

	/// Is the horizontal scroll bar visible?
	bool GetHScrollBar() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETHSCROLLBAR,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Show or hide indentation guides.
	void SetIndentationGuides(bool show) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETINDENTATIONGUIDES,(WPARAM)show,(LPARAM)0);
	}

	/// Are the indentation guides visible?
	bool GetIndentationGuides() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETINDENTATIONGUIDES,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Set the highlighted indentation guide column.
	/// 0 = no highlighted guide.
	void SetHighlightGuide(int column) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHIGHLIGHTGUIDE,(WPARAM)column,(LPARAM)0);
	}

	/// Get the highlighted indentation guide column.
	int GetHighlightGuide() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETHIGHLIGHTGUIDE,(WPARAM)0,(LPARAM)0);
	}

	/// Get the position after the last visible characters on a line.
	int GetLineEndPosition(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINEENDPOSITION,(WPARAM)line,(LPARAM)0);
	}

	/// Get the code page used to interpret the bytes of the document as characters.
	int GetCodePage() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCODEPAGE,(WPARAM)0,(LPARAM)0);
	}

	/// Get the foreground colour of the caret.
	COLORREF GetCaretFore() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (COLORREF)::SendMessage(m_hWnd,SCI_GETCARETFORE,(WPARAM)0,(LPARAM)0);
	}

	/// In palette mode?
	bool GetUsePalette() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETUSEPALETTE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// In read-only mode?
	bool GetReadOnly() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETREADONLY,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Sets the position of the caret.
	void SetCurrentPos(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCURRENTPOS,(WPARAM)pos,(LPARAM)0);
	}

	/// Sets the position that starts the selection - this becomes the anchor.
	void SetSelectionStart(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSELECTIONSTART,(WPARAM)pos,(LPARAM)0);
	}

	/// Returns the position at the start of the selection.
	int GetSelectionStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSELECTIONSTART,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the position that ends the selection - this becomes the currentPosition.
	void SetSelectionEnd(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSELECTIONEND,(WPARAM)pos,(LPARAM)0);
	}

	/// Returns the position at the end of the selection.
	int GetSelectionEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSELECTIONEND,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the print magnification added to the point size of each style for printing.
	void SetPrintMagnification(int magnification) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETPRINTMAGNIFICATION,(WPARAM)magnification,(LPARAM)0);
	}

	/// Returns the print magnification.
	int GetPrintMagnification() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETPRINTMAGNIFICATION,(WPARAM)0,(LPARAM)0);
	}

	/// Modify colours when printing for clearer printed text.
	void SetPrintColourMode(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETPRINTCOLOURMODE,(WPARAM)mode,(LPARAM)0);
	}

	/// Returns the print colour mode.
	int GetPrintColourMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETPRINTCOLOURMODE,(WPARAM)0,(LPARAM)0);
	}

	/// Find some text in the document.
	int FindText(int flags,TextToFind* ft) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_FINDTEXT,(WPARAM)flags,(LPARAM)ft);
	}

	/// On Windows, will draw the document into a display context such as a printer.
	int FormatRange(bool draw,RangeToFormat* fr) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_FORMATRANGE,(WPARAM)draw,(LPARAM)fr);
	}

	/// Retrieve the display line at the top of the display.
	int GetFirstVisibleLine() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETFIRSTVISIBLELINE,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve the contents of a line.
	/// Returns the length of the line.
	int GetLine(int line,char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINE,(WPARAM)line,(LPARAM)text);
	}

	/// Returns the number of lines in the document. There is always at least one.
	int GetLineCount() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINECOUNT,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the size in pixels of the left margin.
	void SetMarginLeft(int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINLEFT,(WPARAM)0,(LPARAM)pixelWidth);
	}

	/// Returns the size in pixels of the left margin.
	int GetMarginLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMARGINLEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the size in pixels of the right margin.
	void SetMarginRight(int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMARGINRIGHT,(WPARAM)0,(LPARAM)pixelWidth);
	}

	/// Returns the size in pixels of the right margin.
	int GetMarginRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMARGINRIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Is the document different from when it was last saved?
	bool GetModify() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETMODIFY,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Select a range of text.
	void SetSel(int start,int end) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSEL,(WPARAM)start,(LPARAM)end);
	}

	/// Retrieve the selected text.
	/// Return the length of the text.
	int GetSelText(char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSELTEXT,(WPARAM)0,(LPARAM)text);
	}

	/// Retrieve a range of text.
	/// Return the length of the text.
	int GetTextRange(TextRange* tr) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTEXTRANGE,(WPARAM)0,(LPARAM)tr);
	}

	/// Draw the selection in normal style or with selection highlighted.
	void HideSelection(bool normal) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HIDESELECTION,(WPARAM)normal,(LPARAM)0);
	}

	/// Retrieve the x value of the point in the window where a position is displayed.
	int PointXFromPosition(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POINTXFROMPOSITION,(WPARAM)0,(LPARAM)pos);
	}

	/// Retrieve the y value of the point in the window where a position is displayed.
	int PointYFromPosition(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POINTYFROMPOSITION,(WPARAM)0,(LPARAM)pos);
	}

	/// Retrieve the line containing a position.
	int LineFromPosition(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_LINEFROMPOSITION,(WPARAM)pos,(LPARAM)0);
	}

	/// Retrieve the position at the start of a line.
	int PositionFromLine(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POSITIONFROMLINE,(WPARAM)line,(LPARAM)0);
	}

	/// Scroll horizontally and vertically.
	void LineScroll(int columns,int lines) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINESCROLL,(WPARAM)columns,(LPARAM)lines);
	}

	/// Ensure the caret is visible.
	void ScrollCaret() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SCROLLCARET,(WPARAM)0,(LPARAM)0);
	}

	/// Replace the selected text with the argument text.
	void ReplaceSel(const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_REPLACESEL,(WPARAM)0,(LPARAM)text);
	}

	/// Set to read only or read write.
	void SetReadOnly(bool readOnly) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETREADONLY,(WPARAM)readOnly,(LPARAM)0);
	}

	/// Null operation.
	void Null() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_NULL,(WPARAM)0,(LPARAM)0);
	}

	/// Will a paste succeed?
	bool CanPaste() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_CANPASTE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Are there any undoable actions in the undo history?
	bool CanUndo() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_CANUNDO,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Delete the undo history.
	void EmptyUndoBuffer() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_EMPTYUNDOBUFFER,(WPARAM)0,(LPARAM)0);
	}

	/// Undo one action in the undo history.
	void Undo() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_UNDO,(WPARAM)0,(LPARAM)0);
	}

	/// Cut the selection to the clipboard.
	void Cut() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CUT,(WPARAM)0,(LPARAM)0);
	}

	/// Copy the selection to the clipboard.
	void Copy() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_COPY,(WPARAM)0,(LPARAM)0);
	}

	/// Paste the contents of the clipboard into the document replacing the selection.
	void Paste() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PASTE,(WPARAM)0,(LPARAM)0);
	}

	/// Clear the selection.
	void Clear() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CLEAR,(WPARAM)0,(LPARAM)0);
	}

	/// Replace the contents of the document with the argument text.
	void SetText(const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTEXT,(WPARAM)0,(LPARAM)text);
	}

	/// Retrieve all the text in the document.
	/// Returns number of characters retrieved.
	int GetText(int length,char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTEXT,(WPARAM)length,(LPARAM)text);
	}

	/// Retrieve the number of characters in the document.
	int GetTextLength() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTEXTLENGTH,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve a pointer to a function that processes messages for this Scintilla.
	int GetDirectFunction() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETDIRECTFUNCTION,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve a pointer value to use as the first argument when calling
	/// the function returned by GetDirectFunction.
	int GetDirectPointer() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETDIRECTPOINTER,(WPARAM)0,(LPARAM)0);
	}

	/// Set to overtype (true) or insert mode.
	void SetOvertype(bool overtype) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETOVERTYPE,(WPARAM)overtype,(LPARAM)0);
	}

	/// Returns true if overtype mode is active otherwise false is returned.
	bool GetOvertype() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETOVERTYPE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Set the width of the insert mode caret.
	void SetCaretWidth(int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCARETWIDTH,(WPARAM)pixelWidth,(LPARAM)0);
	}

	/// Returns the width of the insert mode caret.
	int GetCaretWidth() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCARETWIDTH,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the position that starts the target which is used for updating the
	/// document without affecting the scroll position.
	void SetTargetStart(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTARGETSTART,(WPARAM)pos,(LPARAM)0);
	}

	/// Get the position that starts the target.
	int GetTargetStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTARGETSTART,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the position that ends the target which is used for updating the
	/// document without affecting the scroll position.
	void SetTargetEnd(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTARGETEND,(WPARAM)pos,(LPARAM)0);
	}

	/// Get the position that ends the target.
	int GetTargetEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETTARGETEND,(WPARAM)0,(LPARAM)0);
	}

	/// Replace the target text with the argument text.
	/// Text is counted so it can contain NULs.
	/// Returns the length of the replacement text.
	int ReplaceTarget(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_REPLACETARGET,(WPARAM)length,(LPARAM)text);
	}

	/// Replace the target text with the argument text after \d processing.
	/// Text is counted so it can contain NULs.
	/// Looks for \d where d is between 1 and 9 and replaces these with the strings
	/// matched in the last search operation which were surrounded by \( and \).
	/// Returns the length of the replacement text including any change
	/// caused by processing the \d patterns.
	int ReplaceTargetRE(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_REPLACETARGETRE,(WPARAM)length,(LPARAM)text);
	}

	/// Search for a counted string in the target and set the target to the found
	/// range. Text is counted so it can contain NULs.
	/// Returns length of range or -1 for failure in which case target is not moved.
	int SearchInTarget(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_SEARCHINTARGET,(WPARAM)length,(LPARAM)text);
	}

	/// Set the search flags used by SearchInTarget.
	void SetSearchFlags(int flags) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSEARCHFLAGS,(WPARAM)flags,(LPARAM)0);
	}

	/// Get the search flags used by SearchInTarget.
	int GetSearchFlags() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSEARCHFLAGS,(WPARAM)0,(LPARAM)0);
	}

	/// Show a call tip containing a definition near position pos.
	void CallTipShow(int pos,const char* definition) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPSHOW,(WPARAM)pos,(LPARAM)definition);
	}

	/// Remove the call tip from the screen.
	void CallTipCancel() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPCANCEL,(WPARAM)0,(LPARAM)0);
	}

	/// Is there an active call tip?
	bool CallTipActive() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_CALLTIPACTIVE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Retrieve the position where the caret was before displaying the call tip.
	int CallTipPosStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_CALLTIPPOSSTART,(WPARAM)0,(LPARAM)0);
	}

	/// Highlight a segment of the definition.
	void CallTipSetHlt(int start,int end) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPSETHLT,(WPARAM)start,(LPARAM)end);
	}

	/// Set the background colour for the call tip.
	void CallTipSetBack(COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPSETBACK,(WPARAM)back,(LPARAM)0);
	}

	/// Set the foreground colour for the call tip.
	void CallTipSetFore(COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPSETFORE,(WPARAM)fore,(LPARAM)0);
	}

	/// Set the foreground colour for the highlighted part of the call tip.
	void CallTipSetForeHlt(COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CALLTIPSETFOREHLT,(WPARAM)fore,(LPARAM)0);
	}

	/// Find the display line of a document line taking hidden lines into account.
	int VisibleFromDocLine(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_VISIBLEFROMDOCLINE,(WPARAM)line,(LPARAM)0);
	}

	/// Find the document line of a display line taking hidden lines into account.
	int DocLineFromVisible(int lineDisplay) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_DOCLINEFROMVISIBLE,(WPARAM)lineDisplay,(LPARAM)0);
	}

	/// Set the fold level of a line.
	/// This encodes an integer level along with flags indicating whether the
	/// line is a header and whether it is effectively white space.
	void SetFoldLevel(int line,int level) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOLDLEVEL,(WPARAM)line,(LPARAM)level);
	}

	/// Retrieve the fold level of a line.
	int GetFoldLevel(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETFOLDLEVEL,(WPARAM)line,(LPARAM)0);
	}

	/// Find the last child line of a header line.
	int GetLastChild(int line,int level) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLASTCHILD,(WPARAM)line,(LPARAM)level);
	}

	/// Find the parent line of a child line.
	int GetFoldParent(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETFOLDPARENT,(WPARAM)line,(LPARAM)0);
	}

	/// Make a range of lines visible.
	void ShowLines(int lineStart,int lineEnd) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SHOWLINES,(WPARAM)lineStart,(LPARAM)lineEnd);
	}

	/// Make a range of lines invisible.
	void HideLines(int lineStart,int lineEnd) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HIDELINES,(WPARAM)lineStart,(LPARAM)lineEnd);
	}

	/// Is a line visible?
	bool GetLineVisible(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETLINEVISIBLE,(WPARAM)line,(LPARAM)0)!=0;
	}

	/// Show the children of a header line.
	void SetFoldExpanded(int line,bool expanded) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOLDEXPANDED,(WPARAM)line,(LPARAM)expanded);
	}

	/// Is a header line expanded?
	bool GetFoldExpanded(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETFOLDEXPANDED,(WPARAM)line,(LPARAM)0)!=0;
	}

	/// Switch a header line between expanded and contracted.
	void ToggleFold(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_TOGGLEFOLD,(WPARAM)line,(LPARAM)0);
	}

	/// Ensure a particular line is visible by expanding any header line hiding it.
	void EnsureVisible(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ENSUREVISIBLE,(WPARAM)line,(LPARAM)0);
	}

	/// Set some style options for folding.
	void SetFoldFlags(int flags) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOLDFLAGS,(WPARAM)flags,(LPARAM)0);
	}

	/// Ensure a particular line is visible by expanding any header line hiding it.
	/// Use the currently set visibility policy to determine which range to display.
	void EnsureVisibleEnforcePolicy(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ENSUREVISIBLEENFORCEPOLICY,(WPARAM)line,(LPARAM)0);
	}

	/// Sets whether a tab pressed when caret is within indentation indents.
	void SetTabIndents(bool tabIndents) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTABINDENTS,(WPARAM)tabIndents,(LPARAM)0);
	}

	/// Does a tab pressed when caret is within indentation indent?
	bool GetTabIndents() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETTABINDENTS,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Sets whether a backspace pressed when caret is within indentation unindents.
	void SetBackSpaceUnIndents(bool bsUnIndents) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETBACKSPACEUNINDENTS,(WPARAM)bsUnIndents,(LPARAM)0);
	}

	/// Does a backspace pressed when caret is within indentation unindent?
	bool GetBackSpaceUnIndents() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETBACKSPACEUNINDENTS,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Sets the time the mouse must sit still to generate a mouse dwell event.
	void SetMouseDwellTime(int periodMilliseconds) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMOUSEDWELLTIME,(WPARAM)periodMilliseconds,(LPARAM)0);
	}

	/// Retrieve the time the mouse must sit still to generate a mouse dwell event.
	int GetMouseDwellTime() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMOUSEDWELLTIME,(WPARAM)0,(LPARAM)0);
	}

	/// Get position of start of word.
	int WordStartPosition(int pos,bool onlyWordCharacters) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_WORDSTARTPOSITION,(WPARAM)pos,(LPARAM)onlyWordCharacters);
	}

	/// Get position of end of word.
	int WordEndPosition(int pos,bool onlyWordCharacters) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_WORDENDPOSITION,(WPARAM)pos,(LPARAM)onlyWordCharacters);
	}

	/// Sets whether text is word wrapped.
	void SetWrapMode(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWRAPMODE,(WPARAM)mode,(LPARAM)0);
	}

	/// Retrieve whether text is word wrapped.
	int GetWrapMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETWRAPMODE,(WPARAM)0,(LPARAM)0);
	}

	/// Set the display mode of visual flags for wrapped lines.
	void SetWrapVisualFlags(int wrapVisualFlags) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWRAPVISUALFLAGS,(WPARAM)wrapVisualFlags,(LPARAM)0);
	}

	/// Retrive the display mode of visual flags for wrapped lines.
	int GetWrapVisualFlags() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETWRAPVISUALFLAGS,(WPARAM)0,(LPARAM)0);
	}

	/// Set the location of visual flags for wrapped lines.
	void SetWrapVisualFlagsLocation(int wrapVisualFlagsLocation) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWRAPVISUALFLAGSLOCATION,(WPARAM)wrapVisualFlagsLocation,(LPARAM)0);
	}

	/// Retrive the location of visual flags for wrapped lines.
	int GetWrapVisualFlagsLocation() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETWRAPVISUALFLAGSLOCATION,(WPARAM)0,(LPARAM)0);
	}

	/// Set the start indent for wrapped lines.
	void SetWrapStartIndent(int indent) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWRAPSTARTINDENT,(WPARAM)indent,(LPARAM)0);
	}

	/// Retrive the start indent for wrapped lines.
	int GetWrapStartIndent() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETWRAPSTARTINDENT,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the degree of caching of layout information.
	void SetLayoutCache(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETLAYOUTCACHE,(WPARAM)mode,(LPARAM)0);
	}

	/// Retrieve the degree of caching of layout information.
	int GetLayoutCache() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLAYOUTCACHE,(WPARAM)0,(LPARAM)0);
	}

	/// Sets the document width assumed for scrolling.
	void SetScrollWidth(int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSCROLLWIDTH,(WPARAM)pixelWidth,(LPARAM)0);
	}

	/// Retrieve the document width assumed for scrolling.
	int GetScrollWidth() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSCROLLWIDTH,(WPARAM)0,(LPARAM)0);
	}

	/// Measure the pixel width of some text in a particular style.
	/// NUL terminated text argument.
	/// Does not handle tab or control characters.
	int TextWidth(int style,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_TEXTWIDTH,(WPARAM)style,(LPARAM)text);
	}

	/// Sets the scroll range so that maximum scroll position has
	/// the last line at the bottom of the view (default).
	/// Setting this to false allows scrolling one page below the last line.
	void SetEndAtLastLine(bool endAtLastLine) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETENDATLASTLINE,(WPARAM)endAtLastLine,(LPARAM)0);
	}

	/// Retrieve whether the maximum scroll position has the last
	/// line at the bottom of the view.
	int GetEndAtLastLine() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETENDATLASTLINE,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve the height of a particular line of text in pixels.
	int TextHeight(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_TEXTHEIGHT,(WPARAM)line,(LPARAM)0);
	}

	/// Show or hide the vertical scroll bar.
	void SetVScrollBar(bool show) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETVSCROLLBAR,(WPARAM)show,(LPARAM)0);
	}

	/// Is the vertical scroll bar visible?
	bool GetVScrollBar() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETVSCROLLBAR,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Append a string to the end of the document without changing the selection.
	void AppendText(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_APPENDTEXT,(WPARAM)length,(LPARAM)text);
	}

	/// Is drawing done in two phases with backgrounds drawn before faoregrounds?
	bool GetTwoPhaseDraw() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETTWOPHASEDRAW,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// In twoPhaseDraw mode, drawing is performed in two phases, first the background
	/// and then the foreground. This avoids chopping off characters that overlap the next run.
	void SetTwoPhaseDraw(bool twoPhase) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETTWOPHASEDRAW,(WPARAM)twoPhase,(LPARAM)0);
	}

	/// Make the target range start and end be the same as the selection range start and end.
	void TargetFromSelection() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_TARGETFROMSELECTION,(WPARAM)0,(LPARAM)0);
	}

	/// Join the lines in the target.
	void LinesJoin() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINESJOIN,(WPARAM)0,(LPARAM)0);
	}

	/// Split the lines in the target into lines that are less wide than pixelWidth
	/// where possible.
	void LinesSplit(int pixelWidth) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINESSPLIT,(WPARAM)pixelWidth,(LPARAM)0);
	}

	/// Set the colours used as a chequerboard pattern in the fold margin
	void SetFoldMarginColour(bool useSetting,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOLDMARGINCOLOUR,(WPARAM)useSetting,(LPARAM)back);
	}

	/// Set the colours used as a chequerboard pattern in the fold margin
	void SetFoldMarginHiColour(bool useSetting,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOLDMARGINHICOLOUR,(WPARAM)useSetting,(LPARAM)fore);
	}

	/// Move caret down one line.
	void LineDown() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEDOWN,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret down one line extending selection to new caret position.
	void LineDownExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEDOWNEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret up one line.
	void LineUp() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEUP,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret up one line extending selection to new caret position.
	void LineUpExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEUPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one character.
	void CharLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARLEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one character extending selection to new caret position.
	void CharLeftExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARLEFTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one character.
	void CharRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARRIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one character extending selection to new caret position.
	void CharRightExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARRIGHTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one word.
	void WordLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDLEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one word extending selection to new caret position.
	void WordLeftExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDLEFTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one word.
	void WordRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDRIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one word extending selection to new caret position.
	void WordRightExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDRIGHTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position on line.
	void Home() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOME,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position on line extending selection to new caret position.
	void HomeExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMEEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position on line.
	void LineEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position on line extending selection to new caret position.
	void LineEndExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position in document.
	void DocumentStart() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DOCUMENTSTART,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position in document extending selection to new caret position.
	void DocumentStartExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DOCUMENTSTARTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position in document.
	void DocumentEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DOCUMENTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position in document extending selection to new caret position.
	void DocumentEndExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DOCUMENTENDEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page up.
	void PageUp() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEUP,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page up extending selection to new caret position.
	void PageUpExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEUPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page down.
	void PageDown() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEDOWN,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page down extending selection to new caret position.
	void PageDownExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEDOWNEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Switch from insert to overtype mode or the reverse.
	void EditToggleOvertype() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_EDITTOGGLEOVERTYPE,(WPARAM)0,(LPARAM)0);
	}

	/// Cancel any modes such as call tip or auto-completion list display.
	void Cancel() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CANCEL,(WPARAM)0,(LPARAM)0);
	}

	/// Delete the selection or if no selection, the character before the caret.
	void DeleteBack() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELETEBACK,(WPARAM)0,(LPARAM)0);
	}

	/// If selection is empty or all on one line replace the selection with a tab character.
	/// If more than one line selected, indent the lines.
	void Tab() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_TAB,(WPARAM)0,(LPARAM)0);
	}

	/// Dedent the selected lines.
	void BackTab() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_BACKTAB,(WPARAM)0,(LPARAM)0);
	}

	/// Insert a new line, may use a CRLF, CR or LF depending on EOL mode.
	void NewLine() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_NEWLINE,(WPARAM)0,(LPARAM)0);
	}

	/// Insert a Form Feed character.
	void FormFeed() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_FORMFEED,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to before first visible character on line.
	/// If already there move to first character on line.
	void VCHome() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_VCHOME,(WPARAM)0,(LPARAM)0);
	}

	/// Like VCHome but extending selection to new caret position.
	void VCHomeExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_VCHOMEEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Magnify the displayed text by increasing the sizes by 1 point.
	void ZoomIn() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ZOOMIN,(WPARAM)0,(LPARAM)0);
	}

	/// Make the displayed text smaller by decreasing the sizes by 1 point.
	void ZoomOut() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ZOOMOUT,(WPARAM)0,(LPARAM)0);
	}

	/// Delete the word to the left of the caret.
	void DelWordLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELWORDLEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Delete the word to the right of the caret.
	void DelWordRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELWORDRIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Cut the line containing the caret.
	void LineCut() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINECUT,(WPARAM)0,(LPARAM)0);
	}

	/// Delete the line containing the caret.
	void LineDelete() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEDELETE,(WPARAM)0,(LPARAM)0);
	}

	/// Switch the current line with the previous.
	void LineTranspose() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINETRANSPOSE,(WPARAM)0,(LPARAM)0);
	}

	/// Duplicate the current line.
	void LineDuplicate() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEDUPLICATE,(WPARAM)0,(LPARAM)0);
	}

	/// Transform the selection to lower case.
	void LowerCase() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LOWERCASE,(WPARAM)0,(LPARAM)0);
	}

	/// Transform the selection to upper case.
	void UpperCase() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_UPPERCASE,(WPARAM)0,(LPARAM)0);
	}

	/// Scroll the document down, keeping the caret visible.
	void LineScrollDown() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINESCROLLDOWN,(WPARAM)0,(LPARAM)0);
	}

	/// Scroll the document up, keeping the caret visible.
	void LineScrollUp() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINESCROLLUP,(WPARAM)0,(LPARAM)0);
	}

	/// Delete the selection or if no selection, the character before the caret.
	/// Will not delete the character before at the start of a line.
	void DeleteBackNotLine() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELETEBACKNOTLINE,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position on display line.
	void HomeDisplay() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMEDISPLAY,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position on display line extending selection to
	/// new caret position.
	void HomeDisplayExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMEDISPLAYEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position on display line.
	void LineEndDisplay() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDDISPLAY,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position on display line extending selection to new
	/// caret position.
	void LineEndDisplayExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDDISPLAYEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void HomeWrap() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMEWRAP,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void HomeWrapExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMEWRAPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void LineEndWrap() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDWRAP,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void LineEndWrapExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDWRAPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void VCHomeWrap() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_VCHOMEWRAP,(WPARAM)0,(LPARAM)0);
	}

	/// These are like their namesakes Home(Extend)?, LineEnd(Extend)?, VCHome(Extend)?
	/// except they behave differently when word-wrap is enabled:
	/// They go first to the start / end of the display line, like (Home|LineEnd)Display
	/// The difference is that, the cursor is already at the point, it goes on to the start
	/// or end of the document line, as appropriate for (Home|LineEnd|VCHome)(Extend)?.
	void VCHomeWrapExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_VCHOMEWRAPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Copy the line containing the caret.
	void LineCopy() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINECOPY,(WPARAM)0,(LPARAM)0);
	}

	/// Move the caret inside current view if it's not there already.
	void MoveCaretInsideView() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_MOVECARETINSIDEVIEW,(WPARAM)0,(LPARAM)0);
	}

	/// How many characters are on a line, not including end of line characters?
	int LineLength(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_LINELENGTH,(WPARAM)line,(LPARAM)0);
	}

	/// Highlight the characters at two positions.
	void BraceHighlight(int pos1,int pos2) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_BRACEHIGHLIGHT,(WPARAM)pos1,(LPARAM)pos2);
	}

	/// Highlight the character at a position indicating there is no matching brace.
	void BraceBadLight(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_BRACEBADLIGHT,(WPARAM)pos,(LPARAM)0);
	}

	/// Find the position of a matching brace or INVALID_POSITION if no match.
	int BraceMatch(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_BRACEMATCH,(WPARAM)pos,(LPARAM)0);
	}

	/// Are the end of line characters visible?
	bool GetViewEOL() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETVIEWEOL,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Make the end of line characters visible or invisible.
	void SetViewEOL(bool visible) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETVIEWEOL,(WPARAM)visible,(LPARAM)0);
	}

	/// Retrieve a pointer to the document object.
	int GetDocPointer() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETDOCPOINTER,(WPARAM)0,(LPARAM)0);
	}

	/// Change the document object used.
	void SetDocPointer(int pointer) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETDOCPOINTER,(WPARAM)0,(LPARAM)pointer);
	}

	/// Set which document modification events are sent to the container.
	void SetModEventMask(int mask) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMODEVENTMASK,(WPARAM)mask,(LPARAM)0);
	}

	/// Retrieve the column number which text should be kept within.
	int GetEdgeColumn() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETEDGECOLUMN,(WPARAM)0,(LPARAM)0);
	}

	/// Set the column number of the edge.
	/// If text goes past the edge then it is highlighted.
	void SetEdgeColumn(int column) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETEDGECOLUMN,(WPARAM)column,(LPARAM)0);
	}

	/// Retrieve the edge highlight mode.
	int GetEdgeMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETEDGEMODE,(WPARAM)0,(LPARAM)0);
	}

	/// The edge may be displayed by a line (EDGE_LINE) or by highlighting text that
	/// goes beyond it (EDGE_BACKGROUND) or not displayed at all (EDGE_NONE).
	void SetEdgeMode(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETEDGEMODE,(WPARAM)mode,(LPARAM)0);
	}

	/// Retrieve the colour used in edge indication.
	COLORREF GetEdgeColour() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (COLORREF)::SendMessage(m_hWnd,SCI_GETEDGECOLOUR,(WPARAM)0,(LPARAM)0);
	}

	/// Change the colour used in edge indication.
	void SetEdgeColour(COLORREF edgeColour) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETEDGECOLOUR,(WPARAM)edgeColour,(LPARAM)0);
	}

	/// Sets the current caret position to be the search anchor.
	void SearchAnchor() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SEARCHANCHOR,(WPARAM)0,(LPARAM)0);
	}

	/// Find some text starting at the search anchor.
	/// Does not ensure the selection is visible.
	int SearchNext(int flags,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_SEARCHNEXT,(WPARAM)flags,(LPARAM)text);
	}

	/// Find some text starting at the search anchor and moving backwards.
	/// Does not ensure the selection is visible.
	int SearchPrev(int flags,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_SEARCHPREV,(WPARAM)flags,(LPARAM)text);
	}

	/// Retrieves the number of lines completely visible.
	int LinesOnScreen() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_LINESONSCREEN,(WPARAM)0,(LPARAM)0);
	}

	/// Set whether a pop up menu is displayed automatically when the user presses
	/// the wrong mouse button.
	void UsePopUp(bool allowPopUp) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_USEPOPUP,(WPARAM)allowPopUp,(LPARAM)0);
	}

	/// Is the selection rectangular? The alternative is the more common stream selection.
	bool SelectionIsRectangle() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_SELECTIONISRECTANGLE,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Set the zoom level. This number of points is added to the size of all fonts.
	/// It may be positive to magnify or negative to reduce.
	void SetZoom(int zoom) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETZOOM,(WPARAM)zoom,(LPARAM)0);
	}

	/// Retrieve the zoom level.
	int GetZoom() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETZOOM,(WPARAM)0,(LPARAM)0);
	}

	/// Create a new document object.
	/// Starts with reference count of 1 and not selected into editor.
	int CreateDocument() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_CREATEDOCUMENT,(WPARAM)0,(LPARAM)0);
	}

	/// Extend life of document.
	void AddRefDocument(int doc) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ADDREFDOCUMENT,(WPARAM)0,(LPARAM)doc);
	}

	/// Release a reference to the document, deleting document if it fades to black.
	void ReleaseDocument(int doc) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_RELEASEDOCUMENT,(WPARAM)0,(LPARAM)doc);
	}

	/// Get which document modification events are sent to the container.
	int GetModEventMask() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETMODEVENTMASK,(WPARAM)0,(LPARAM)0);
	}

	/// Change internal focus flag.
	void SetFocus(bool focus) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETFOCUS,(WPARAM)focus,(LPARAM)0);
	}

	/// Get internal focus flag.
	bool GetFocus() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETFOCUS,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Change error status - 0 = OK.
	void SetStatus(int statusCode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSTATUS,(WPARAM)statusCode,(LPARAM)0);
	}

	/// Get error status.
	int GetStatus() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSTATUS,(WPARAM)0,(LPARAM)0);
	}

	/// Set whether the mouse is captured when its button is pressed.
	void SetMouseDownCaptures(bool captures) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETMOUSEDOWNCAPTURES,(WPARAM)captures,(LPARAM)0);
	}

	/// Get whether mouse gets captured.
	bool GetMouseDownCaptures() {
		ATLASSERT(::IsWindow(m_hWnd));
		return ::SendMessage(m_hWnd,SCI_GETMOUSEDOWNCAPTURES,(WPARAM)0,(LPARAM)0)!=0;
	}

	/// Sets the cursor to one of the SC_CURSOR* values.
	void SetCursor(int cursorType) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCURSOR,(WPARAM)cursorType,(LPARAM)0);
	}

	/// Get cursor type.
	int GetCursor() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCURSOR,(WPARAM)0,(LPARAM)0);
	}

	/// Change the way control characters are displayed:
	/// If symbol is < 32, keep the drawn way, else, use the given character.
	void SetControlCharSymbol(int symbol) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCONTROLCHARSYMBOL,(WPARAM)symbol,(LPARAM)0);
	}

	/// Get the way control characters are displayed.
	int GetControlCharSymbol() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETCONTROLCHARSYMBOL,(WPARAM)0,(LPARAM)0);
	}

	/// Move to the previous change in capitalisation.
	void WordPartLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDPARTLEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Move to the previous change in capitalisation extending selection
	/// to new caret position.
	void WordPartLeftExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDPARTLEFTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move to the change next in capitalisation.
	void WordPartRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDPARTRIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Move to the next change in capitalisation extending selection
	/// to new caret position.
	void WordPartRightExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDPARTRIGHTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Set the way the display area is determined when a particular line
	/// is to be moved to by Find, FindNext, GotoLine, etc.
	void SetVisiblePolicy(int visiblePolicy,int visibleSlop) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETVISIBLEPOLICY,(WPARAM)visiblePolicy,(LPARAM)visibleSlop);
	}

	/// Delete back from the current position to the start of the line.
	void DelLineLeft() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELLINELEFT,(WPARAM)0,(LPARAM)0);
	}

	/// Delete forwards from the current position to the end of the line.
	void DelLineRight() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_DELLINERIGHT,(WPARAM)0,(LPARAM)0);
	}

	/// Get and Set the xOffset (ie, horizonal scroll position).
	void SetXOffset(int newOffset) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETXOFFSET,(WPARAM)newOffset,(LPARAM)0);
	}

	/// Get and Set the xOffset (ie, horizonal scroll position).
	int GetXOffset() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETXOFFSET,(WPARAM)0,(LPARAM)0);
	}

	/// Set the last x chosen value to be the caret x position.
	void ChooseCaretX() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHOOSECARETX,(WPARAM)0,(LPARAM)0);
	}

	/// Set the focus to this Scintilla widget.
	/// GTK+ Specific.
	void GrabFocus() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_GRABFOCUS,(WPARAM)0,(LPARAM)0);
	}

	/// Set the way the caret is kept visible when going sideway.
	/// The exclusion zone is given in pixels.
	void SetXCaretPolicy(int caretPolicy,int caretSlop) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETXCARETPOLICY,(WPARAM)caretPolicy,(LPARAM)caretSlop);
	}

	/// Set the way the line the caret is on is kept visible.
	/// The exclusion zone is given in lines.
	void SetYCaretPolicy(int caretPolicy,int caretSlop) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETYCARETPOLICY,(WPARAM)caretPolicy,(LPARAM)caretSlop);
	}

	/// Set printing to line wrapped (SC_WRAP_WORD) or not line wrapped (SC_WRAP_NONE).
	void SetPrintWrapMode(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETPRINTWRAPMODE,(WPARAM)mode,(LPARAM)0);
	}

	/// Is printing line wrapped?
	int GetPrintWrapMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETPRINTWRAPMODE,(WPARAM)0,(LPARAM)0);
	}

	/// Set a fore colour for active hotspots.
	void SetHotspotActiveFore(bool useSetting,COLORREF fore) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHOTSPOTACTIVEFORE,(WPARAM)useSetting,(LPARAM)fore);
	}

	/// Set a back colour for active hotspots.
	void SetHotspotActiveBack(bool useSetting,COLORREF back) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHOTSPOTACTIVEBACK,(WPARAM)useSetting,(LPARAM)back);
	}

	/// Enable / Disable underlining active hotspots.
	void SetHotspotActiveUnderline(bool underline) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHOTSPOTACTIVEUNDERLINE,(WPARAM)underline,(LPARAM)0);
	}

	/// Limit hotspots to single line so hotspots on two lines don't merge.
	void SetHotspotSingleLine(bool singleLine) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETHOTSPOTSINGLELINE,(WPARAM)singleLine,(LPARAM)0);
	}

	/// Move caret between paragraphs (delimited by empty lines).
	void ParaDown() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PARADOWN,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret between paragraphs (delimited by empty lines).
	void ParaDownExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PARADOWNEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret between paragraphs (delimited by empty lines).
	void ParaUp() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PARAUP,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret between paragraphs (delimited by empty lines).
	void ParaUpExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PARAUPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Given a valid document position, return the previous position taking code
	/// page into account. Returns 0 if passed 0.
	int PositionBefore(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POSITIONBEFORE,(WPARAM)pos,(LPARAM)0);
	}

	/// Given a valid document position, return the next position taking code
	/// page into account. Maximum value returned is the last position in the document.
	int PositionAfter(int pos) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_POSITIONAFTER,(WPARAM)pos,(LPARAM)0);
	}

	/// Copy a range of text to the clipboard. Positions are clipped into the document.
	void CopyRange(int start,int end) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_COPYRANGE,(WPARAM)start,(LPARAM)end);
	}

	/// Copy argument text to the clipboard.
	void CopyText(int length,const char* text) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_COPYTEXT,(WPARAM)length,(LPARAM)text);
	}

	/// Set the selection mode to stream (SC_SEL_STREAM) or rectangular (SC_SEL_RECTANGLE) or
	/// by lines (SC_SEL_LINES).
	void SetSelectionMode(int mode) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETSELECTIONMODE,(WPARAM)mode,(LPARAM)0);
	}

	/// Get the mode of the current selection.
	int GetSelectionMode() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETSELECTIONMODE,(WPARAM)0,(LPARAM)0);
	}

	/// Retrieve the position of the start of the selection at the given line (INVALID_POSITION if no selection on this line).
	int GetLineSelStartPosition(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINESELSTARTPOSITION,(WPARAM)line,(LPARAM)0);
	}

	/// Retrieve the position of the end of the selection at the given line (INVALID_POSITION if no selection on this line).
	int GetLineSelEndPosition(int line) {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLINESELENDPOSITION,(WPARAM)line,(LPARAM)0);
	}

	/// Move caret down one line, extending rectangular selection to new caret position.
	void LineDownRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEDOWNRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret up one line, extending rectangular selection to new caret position.
	void LineUpRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEUPRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one character, extending rectangular selection to new caret position.
	void CharLeftRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARLEFTRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one character, extending rectangular selection to new caret position.
	void CharRightRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_CHARRIGHTRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to first position on line, extending rectangular selection to new caret position.
	void HomeRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_HOMERECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to before first visible character on line.
	/// If already there move to first character on line.
	/// In either case, extend rectangular selection to new caret position.
	void VCHomeRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_VCHOMERECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to last position on line, extending rectangular selection to new caret position.
	void LineEndRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LINEENDRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page up, extending rectangular selection to new caret position.
	void PageUpRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEUPRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret one page down, extending rectangular selection to new caret position.
	void PageDownRectExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_PAGEDOWNRECTEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to top of page, or one page up if already at top of page.
	void StutteredPageUp() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STUTTEREDPAGEUP,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to top of page, or one page up if already at top of page, extending selection to new caret position.
	void StutteredPageUpExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STUTTEREDPAGEUPEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to bottom of page, or one page down if already at bottom of page.
	void StutteredPageDown() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STUTTEREDPAGEDOWN,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret to bottom of page, or one page down if already at bottom of page, extending selection to new caret position.
	void StutteredPageDownExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STUTTEREDPAGEDOWNEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one word, position cursor at end of word.
	void WordLeftEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDLEFTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret left one word, position cursor at end of word, extending selection to new caret position.
	void WordLeftEndExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDLEFTENDEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one word, position cursor at end of word.
	void WordRightEnd() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDRIGHTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Move caret right one word, position cursor at end of word, extending selection to new caret position.
	void WordRightEndExtend() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_WORDRIGHTENDEXTEND,(WPARAM)0,(LPARAM)0);
	}

	/// Set the set of characters making up whitespace for when moving or selecting by word.
	/// Should be called after SetWordChars.
	void SetWhitespaceChars(const char* characters) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETWHITESPACECHARS,(WPARAM)0,(LPARAM)characters);
	}

	/// Reset the set of characters for whitespace and word characters to the defaults.
	void SetCharsDefault() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETCHARSDEFAULT,(WPARAM)0,(LPARAM)0);
	}

	/// Get currently selected item position in the auto-completion list
	int AutoCGetCurrent() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_AUTOCGETCURRENT,(WPARAM)0,(LPARAM)0);
	}

	/// Enlarge the document to a particular size of text bytes.
	void Allocate(int bytes) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_ALLOCATE,(WPARAM)bytes,(LPARAM)0);
	}

	/// Start notifying the container of all key presses and commands.
	void StartRecord() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STARTRECORD,(WPARAM)0,(LPARAM)0);
	}

	/// Stop notifying the container of all key presses and commands.
	void StopRecord() {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_STOPRECORD,(WPARAM)0,(LPARAM)0);
	}

	/// Set the lexing language of the document.
	void SetLexer(int lexer) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETLEXER,(WPARAM)lexer,(LPARAM)0);
	}

	/// Retrieve the lexing language of the document.
	int GetLexer() {
		ATLASSERT(::IsWindow(m_hWnd));
		return (int)::SendMessage(m_hWnd,SCI_GETLEXER,(WPARAM)0,(LPARAM)0);
	}

	/// Colourise a segment of the document using the current lexing language.
	void Colourise(int start,int end) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_COLOURISE,(WPARAM)start,(LPARAM)end);
	}

	/// Set up a value that may be used by a lexer for some optional feature.
	void SetProperty(const char* key,const char* value) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETPROPERTY,(WPARAM)key,(LPARAM)value);
	}

	/// Set up the key words used by the lexer.
	void SetKeyWords(int keywordSet,const char* keyWords) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETKEYWORDS,(WPARAM)keywordSet,(LPARAM)keyWords);
	}

	/// Set the lexing language of the document based on string name.
	void SetLexerLanguage(const char* language) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_SETLEXERLANGUAGE,(WPARAM)0,(LPARAM)language);
	}

	/// Load a lexer library (dll / so).
	void LoadLexerLibrary(const char* path) {
		ATLASSERT(::IsWindow(m_hWnd));
		::SendMessage(m_hWnd,SCI_LOADLEXERLIBRARY,(WPARAM)0,(LPARAM)path);
	}

	//--Autogenerated
};

typedef CScintillaT<CWindow>		CScintilla;

}; //namespace WTL

#endif // __ATLSCINTILLA_H__
