//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
//
// Smart handle class
//
// SmrtHandle implements a template class for dealing with all kinds
// of handles. It automatically closes the encapsulated handle
// using the appropriate function. A reference count is used to 
// keep track of the number of copies referring to a given handle.
//
//------------------------------------------------------------------
//
// General usage:
// -------------
//
// Pass the raw handle to the constructor. Alternatively, use the
// reference operator to assign a raw handle. 
// Note that you are not allowed to asign a raw handle to SmrtHandle:
//
//  WRONG:  
//    util::smrthandle::SmrtFileHandle hFile;
//    hFile = CreateFile(...);
//
//  RIGHT:  
//    util::smrthandle::SmrtFileHandle hFile(CreateFile(...));
//
//  RIGHT:
//    util::smrthandle::SmrtRegHandle hReg;
//    RegOpenKey(..., &hReg);
//
//  RIGHT:
//    util::smrthandle::SmrtFileHandle hFile1(CreateFile(...));
//    util::smrthandle::SmrtFileHandle hFile2;
//    ...
//    hFile2 = hFile1; // assignment between SmartHandles allowed
//
// SmrtHandle::isNull()
// --------------------

// SmrtHandle defines a member function isNull(), which you should
// use to test the handle against the "NULL" handle value. In fact, 
// some handle types don't use NULL (= 0), but some other value to
// represent a NULL (invalid) handle. Eg: INVALID_HANDLE_VALUE returned
// by CreateFile().
//
//
// Predefined specializations:
// --------------------------
// 
// Depending on the headers included BEFORE "SmrtHandle.h", the following
// specialized smart handles are defined:
//
//  Type                  Typically used with...
//  ----                  ----------------------
//  SmrtWin32Handle     : general Win32 handle, closes with CloseHandle()
//  SmrtFileMap         : MapViewOfFile() & friends
//  SmrtLibHandle       : LoadLibrary()
//  SmrtFindHandle      : FindFirstFile()
//  SmrtFileHandle      : CreateFile()
//  SmrtRegHandle       : RegOpenKeyEx()
//  SmrtSafeArrayLock   : SafeArrayAccessData() 
//  SmrtCoTaskMemPtr    : CoTaskMemAlloc()
//  SmrtMsiHandle       : MsiOpenDatabase()
//  SmrtHInternet       : InternetConnect()
//  SmrtCFileHandle     : fopen()
//  SmrtCTempFileHandle : holds a pointer to a filename, and deletes the
//                        file when the smart handle is destroyed.
//
// Defining your own specializations:
// ---------------------------------
// 
// To create your own smart handle specializations, you need to know:
//  SomeHandle: the handle type
//  SomeNull:   the value of the "NULL" handle
//  SomeClose:  the function used to close the handle
//  SomeReturn: ... that function's return type
//  SomeDecl:   ... that function's calling convention
// 
// Then define SomeSmrtHandle as :
// 
//  struct SomeNullHandle { static Handle null() { return SomeNull; } };
//  typedef util::smrthandle::SmrtHandle< SomeDecl < SomeHandle, SomeReturn, SomeClose>, 
//                          SomeNullHandle >
//          SomeSmrtHandle;
// 
// This header defines two calling convention policy classes to be used
// as "SomeDecl": 
//
//  - StdCall, to be used for __stdcall functions
//  - CDecl,   to be used for __cdecl functions
//
// If your "NULL" handle happens to be 0/NULL, it also predefines the
// struct NullHandle, which is the last template argument's default parameter.

// Let's create a smart handle for handles returned by the win32
// function FindFirstFile().
//
//  SomeHandle is HANDLE
//  SomeNull   is INVALID_HANDLE_VALUE (not NULL !!)
//  SomeClose  is ::CloseHandle()
//  SomeReturn is BOOL
//  SomeDecl   is __stdcall (like all Win32 functions)
//
//  struct InvalidHandle { static HANDLE null() { return INVALID_HANDLE_VALUE; } };
//  typedef SmrtHandle<StdCall<HANDLE, BOOL, ::FindClose>, InvalidHandle >
//          SmrtFindHandle;
//
// Experts:
// -------
// Instead of using StdCall or CDecl, you can define any other struct
// with a close() member function. This allows you to perform an 
// arbitrary complex operation to close a handle, use functions
// with more than one parameter, ... 
// 
//------------------------------------------------------------------
#ifndef CAMIT_SMRT_HANDLE_H
#define CAMIT_SMRT_HANDLE_H

#include "compiler.h"

namespace camit {

// Default Null handle for value "0"
template <typename Handle>
struct NullHandle { static Handle null() { return 0; } };

// "Policy" for __stdcall function types
template <typename H, typename R, R (__stdcall* func)(H)>
struct StdCall 
{ 
  typedef H  Handle;
  typedef H* HandlePointer;
  typedef R  ReturnType;
  static ReturnType close(Handle hndl) { return func(hndl); } 
};

// "Policy" for __stdcall function types where the close function takes a casted handle
template <typename H, typename H2, typename R, R (__stdcall* func)(H2)>
struct StdCall2 
{ 
  typedef H  Handle;
  typedef H* HandlePointer;
  typedef R  ReturnType;
  static ReturnType close(Handle hndl) { return func(static_cast<H2>(hndl)); } 
};

// "Policy" for __cdecl function types
template <typename H, typename R, R (__cdecl* func)(H)>
struct CDecl 
{ 
  typedef H  Handle;
  typedef H* HandlePointer;
  typedef R  ReturnType;
  static ReturnType close(Handle hndl) { return func(hndl); } 
};

// "Policy" for __cdecl function types where the close function takes a casted handle
template <typename H, typename H2, typename R, R (__cdecl* func)(H2)>
struct CDecl2 
{ 
  typedef H  Handle;
  typedef H* HandlePointer;
  typedef R  ReturnType;
  static ReturnType close(Handle hndl) { return func(static_cast<H2>(hndl)); } 
};


// SmrtHandle class
template <
          typename C, 
          typename N = NullHandle<CAMIT_TYPENAME C::Handle> 
         >
class SmrtHandle : public C
{
public:
  typedef typename C::Handle Handle;
  typedef typename C::HandlePointer HandlePointer;
  typedef typename C::ReturnType ReturnType;

  // constructor
  explicit SmrtHandle(const Handle& hndl = N::null())
    : m_hndl(hndl) 
  {
    try {
      m_pcnt = new long(1);
    }
    catch (...) {
      close(hndl);
      throw;
    }
  };

  // copy constructor
  SmrtHandle(const SmrtHandle& rhs)
  : m_hndl(rhs.m_hndl),
    m_pcnt(rhs.m_pcnt)
  {
    ++(*m_pcnt);
  };

  // destructor
  ~SmrtHandle() 
  { 
    destroy();
  }
  
  // assignment to smart handle
  SmrtHandle& operator=(const SmrtHandle& rhs)
  { 
    if (m_pcnt != rhs.m_pcnt) {
      ++(*rhs.m_pcnt);
      destroy();
      m_hndl = rhs.m_hndl;
      m_pcnt = rhs.m_pcnt;      
    }
    
    return (*this); 
  }

  // reset smart handle (closes owned handle)
  void reset(const Handle& hndl = N::null())
  {   
    if (--(*m_pcnt) == 0) {
      close(m_hndl);
    }
    else {
      try {
        m_pcnt = new long;
      }
      catch (...) {
        close(hndl);
        throw;
      }
    }

    m_hndl = hndl;
    *m_pcnt = 1;
  }

  // release and return encapsulated handle
  Handle release() 
  { 
    if (*m_pcnt > 1) {
      long* ptmp = new long(1);
      --(*m_pcnt);
      m_pcnt = ptmp;
    }

    Handle tmp = m_hndl;
    m_hndl = N::null(); // set encapsulated handle to NULL
    return tmp; 
  }

  // reference operator (destroys enclosed handle)
  HandlePointer operator&()
  { 
    reset();
    return &m_hndl; 
  }

  // return encaplsulated handle
  const Handle & get() const            { return m_hndl; }

  operator Handle()                     {  return m_hndl; }

//  operator const Handle() const         {  return m_hndl; }

  // comparison
  bool equals(const Handle& hndl) const { return m_hndl == hndl; }

  // check if handle is valid
  bool isNull() const { return m_hndl == N::null(); }

private:
   void destroy() 
   { 
     if (--(*m_pcnt) == 0) { 
       close(m_hndl);
       delete m_pcnt;
     }
   }

   void close(Handle hndl) {
     if (hndl != N::null()) {
       C::close(hndl);
     }
   }

protected:
  Handle        m_hndl;     // encapsulated handle
  long*         m_pcnt;     // pointer to reference counter
};

//------------------------------------------------------------------------------
// SmrtHandle version that maintains a value passed as the first argument
// to the destroy function.
//------------------------------------------------------------------------------
template <typename A, typename H, typename R, R (__stdcall* func)(A, H)>
struct StdCallArg 
{ 
  typedef H  Handle;
  typedef H* HandlePointer;
  typedef R  ReturnType;
  typedef A  ArgType;
  ReturnType close(Handle hndl) { return func(arg_, hndl); } 
  ArgType arg_;
};

#if defined(_MSC_VER) && _MSC_VER <= 1200

template <typename A, typename H, typename C, typename N = NullHandle<H> >
class SmrtArgHandle : public SmrtHandle<C, N>
{
public:
  typedef A  ArgType;
  typedef H  Handle;
  
  SmrtArgHandle(const ArgType& arg = ArgType(), const Handle& hndl = N::null()) :
    SmrtHandle<C, N>(hndl)
  {
    arg_ = arg;
  }

  SmrtArgHandle(const SmrtArgHandle& rhs) :
    SmrtHandle<C, N>(static_cast<const SmrtHandle&>(rhs))
  {
    arg_ = rhs.arg_;
  }

 void reset(const ArgType& arg = ArgType(), const Handle& hndl = N::null())
 {
   arg_ = arg;
   SmrtHandle<C, N>::reset(hndl);
 }

  operator Handle() {  return m_hndl; }
  operator const Handle() const {  return m_hndl; }

};

#else // !(_MSC_VER <= 1200)

template <typename C, typename N = NullHandle<CAMIT_TYPENAME C::Handle> >
class SmrtArgHandle : public SmrtHandle<C, N>
{
public:
  typedef typename C::ArgType ArgType;
  typedef typename C::Handle Handle;
  typedef SmrtHandle<C, N> Base;
  
  SmrtArgHandle(const ArgType& arg = ArgType(), const Handle& hndl = N::null()) :
    Base(hndl)
  {
    arg_ = arg;
  }

  SmrtArgHandle(const SmrtArgHandle& rhs) :
    Base(static_cast<const Base&>(rhs))
  {
    arg_ = rhs.arg_;
  }

 void reset(const ArgType& arg = ArgType(), const Handle& hndl = N::null())
 {
   arg_ = arg;
   Base::reset(hndl);
 }
};

#endif // !(_MSC_VER <= 1200)



//-------------------------------------------------------------------------
// #include <windows.h>
//-------------------------------------------------------------------------
#if defined(_WINBASE_) || defined(_WINBASE_H)
// file related handles use INVALID_HANDLE_VALUE as the "null" handle !
struct InvalidHandle { static HANDLE null() { return INVALID_HANDLE_VALUE; } };

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// IF YOU ARE GETTING COMPILER ERRORS, AND YOU ARE USING VISUAL C++ 6, MAKE
// SURE YOU INSTALLED THE LATEST SERVICE PACK FROM
// http://msdn.microsoft.com/vstudio/downloads/updates/sp/vs6/sp5/default.aspx !
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
typedef SmrtHandle<StdCall<HANDLE, BOOL, ::CloseHandle> >            
        SmrtWin32Handle;

typedef SmrtHandle<StdCall<HINSTANCE, BOOL, ::FreeLibrary> >
        SmrtLibHandle;

typedef SmrtHandle<StdCall<HANDLE, BOOL, ::FindClose>, InvalidHandle >
        SmrtFindHandle;

typedef SmrtHandle<StdCall<HANDLE, BOOL, ::CloseHandle>, InvalidHandle >
        SmrtFileHandle;

typedef SmrtHandle<StdCall<HDC, BOOL, ::DeleteDC> >
        SmrtDcHandle;

typedef SmrtHandle<StdCall<HWND, BOOL, ::DestroyWindow> >
        SmrtWndHandle;

typedef SmrtHandle<StdCall<HHOOK, BOOL, ::UnhookWindowsHookEx> >
        SmrtWindowsHookHandle;

typedef SmrtHandle<StdCall<HICON, BOOL, ::DestroyIcon> >
        SmrtIconHandle;

typedef SmrtHandle<StdCall<HCURSOR, BOOL, ::DestroyCursor> >
        SmrtCursorHandle;

typedef SmrtHandle<StdCall<ATOM, ATOM, ::GlobalDeleteAtom> >
        SmrtAtomHandle;

// convenience typedef for GDI handles
template <typename H>
struct SmrtGdiHandleTempl {
  typedef SmrtHandle<StdCall2<H, HGDIOBJ, BOOL, ::DeleteObject> > Type;
};

typedef SmrtGdiHandleTempl<HGDIOBJ>::Type  SmrtGdiHandle;
typedef SmrtGdiHandleTempl<HRGN>::Type     SmrtRgnHandle;
typedef SmrtGdiHandleTempl<HBRUSH>::Type   SmrtBrushHandle;
typedef SmrtGdiHandleTempl<HPEN>::Type     SmrtPenHandle;
typedef SmrtGdiHandleTempl<HFONT>::Type    SmrtFontHandle;
typedef SmrtGdiHandleTempl<HBITMAP>::Type  SmrtBitMapHandle;


#if defined(_MSC_VER) && _MSC_VER <= 1200

typedef SmrtArgHandle<HWND, HDC, StdCallArg<HWND, HDC, int, ::ReleaseDC> >           SmrtDcRelease;
typedef SmrtArgHandle<HDC, HGDIOBJ, StdCallArg<HDC, HGDIOBJ, HGDIOBJ, ::SelectObject> > SmrtGdiSelect;
typedef SmrtArgHandle<HWND, int, StdCallArg<HWND, int, BOOL, ::UnregisterHotKey> >   SmrtHotKey;

#else 

typedef SmrtArgHandle<StdCallArg<HWND, HDC, int, ::ReleaseDC> >           SmrtDcRelease;
typedef SmrtArgHandle<StdCallArg<HDC, HGDIOBJ, HGDIOBJ, ::SelectObject> > SmrtGdiSelect;
typedef SmrtArgHandle<StdCallArg<HWND, int, BOOL, ::UnregisterHotKey> >   SmrtHotKey;

#endif


#endif // _WINBASE_

//-------------------------------------------------------------------------
// #include <Winreg.h>
//-------------------------------------------------------------------------
#if defined(_WINREG_) || defined(_WINREG_H)
typedef SmrtHandle<StdCall<HKEY, LONG, ::RegCloseKey> >
        SmrtRegHandle;
#endif // _WINREG_

//-------------------------------------------------------------------------
// #include <oleauto.h>
//-------------------------------------------------------------------------
#if defined(_OLEAUTO_H_) || defined(_OLEAUTO_H)
typedef SmrtHandle<StdCall<SAFEARRAY*,HRESULT, ::SafeArrayUnaccessData> >
        SmrtSafeArrayLock;
#endif // _OLEAUTO_H_

//-------------------------------------------------------------------------
// #include <objbase.h>
//-------------------------------------------------------------------------
#if defined(_OBJBASE_H_) || defined(_OBJBASE_H)
typedef SmrtHandle<StdCall<LPVOID, void, ::CoTaskMemFree> >
        SmrtCoTaskMemPtr;
#endif // _OBJBASE_H_

//-------------------------------------------------------------------------
// #include <Msi.h>
//-------------------------------------------------------------------------
#if defined(_WIN32_MSI)
typedef SmrtHandle<StdCall<MSIHANDLE, UINT, ::MsiCloseHandle> >
        SmrtMsiHandle;
#endif // _WIN32_MSI

//-------------------------------------------------------------------------
// #include <Wininet.h>
//-------------------------------------------------------------------------
#if defined(_WININET_) || defined(_WININET_H)
typedef SmrtHandle<StdCall<HINTERNET, BOOL, ::InternetCloseHandle> >
        SmrtHInternet;
#endif // _WININET_

//-------------------------------------------------------------------------
// #include <stdio.h>
//-------------------------------------------------------------------------
#if defined(stdin) || defined(_STDIO_H_)
// SmrtCFileHandle: stdio file handle
typedef SmrtHandle<CDecl<FILE*, int, ::fclose> >
        SmrtCFileHandle;

// SmrtCTempFileHandle: calls unlink() upon destruction
typedef SmrtHandle<CDecl<const char*, int, ::remove> >
        SmrtCTempFileHandle;
#endif // stdin

} // namespace camit

#endif // CAMIT_SMRT_HANDLE_H


