//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "Console.h"
#include "MutexLock.h"
#include <winerror.h>
#include <process.h>
#include <algorithm>
#include <functional>
#include <stdlib.h>

namespace camit {

#define BUFFER_SIZE     160

//------------------------------------------------------------------------------
Console::Console() :
    usedBuffer_(0),
    maxCmdLength_(0)
{
    // allocate input buffer
    buffer_ = new char[BUFFER_SIZE];
}

//------------------------------------------------------------------------------
Console::~Console()
{
    stop();
    delete [] buffer_;
}

//------------------------------------------------------------------------------
void Console::registerCommand(const std::string& cmd, int id, UINT hotkey)
{
    Command command;
    command.threadId = GetCurrentThreadId();
    command.cmd      = cmd + "!";
    command.id       = id;
    command.hotkey   = hotkey;

    // change to lower case
    std::transform(command.cmd.begin(), command.cmd.end(), command.cmd.begin(), std::ptr_fun(tolower));

    commands_.push_front(command);

    if (cmd.length() > maxCmdLength_) 
    {
        maxCmdLength_ = cmd.length();
    }
}

//------------------------------------------------------------------------------
void Console::start()
{
    // only start if thread stopped
    if (WaitForSingleObject(thread_, 0) == WAIT_TIMEOUT)
        return;

    unsigned id;
    HANDLE res = reinterpret_cast<HANDLE>(_beginthreadex(NULL, 0, threadProcStub, reinterpret_cast<LPVOID>(this), CREATE_SUSPENDED, &id));
    thread_.reset(res);
    ResumeThread(thread_);
}

//------------------------------------------------------------------------------
void Console::stop()
{
    // we have to forcibly terminate the console thread, as there is no way
    // to do non-blocking console I/O under Win32!
    TerminateThread(thread_, 0);
}

//------------------------------------------------------------------------------
unsigned int WINAPI Console::threadProcStub(LPVOID lp)
{
    return reinterpret_cast<Console*>(lp)->threadProc();
}

//------------------------------------------------------------------------------
unsigned int Console::threadProc()
{
    while (true) 
    {
        TCHAR c = getchar();

        if (c != EOF && isprint(c)) 
        {
            parseCommand(c);
        }
    }

    return 0;
}

//------------------------------------------------------------------------------
bool Console::parseCommand(char c)
{
    bool b = false;
    buffer_[usedBuffer_++] = tolower(c);
    buffer_[usedBuffer_] = '\0';

    // find first exclamation mark
    char* eoc = strchr(buffer_, _T('!'));
    if (!eoc) 
    {
        if (usedBuffer_ > maxCmdLength_) 
        {
            memmove(buffer_, buffer_ + usedBuffer_ - maxCmdLength_, usedBuffer_ - maxCmdLength_);
        }
        return false;
    }
    ++eoc;

    for (Commands::const_iterator it = commands_.begin(); it != commands_.end(); ++it) 
    {

        if (eoc - buffer_ >= static_cast<int>(it->cmd.length()) && it->cmd == (eoc - it->cmd.length())) 
        {

            PostThreadMessage(it->threadId, WM_HOTKEY, static_cast<WPARAM>(it->id), static_cast<LPARAM>(it->hotkey));
            b = true;

        }
    }

    memmove(buffer_, eoc, usedBuffer_ - (eoc - buffer_));
    usedBuffer_ -= (eoc - buffer_);
    return b;
}

} // namespace camit
