//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "FrameGrabber.h"
#include "FrameWnd.h"
#include "ArgsInfo.h"
#include "Grabber.h"
#include "../hook/hook.h"
#include <process.h>
#include <algorithm>

namespace camit {

//------------------------------------------------------------------------------
FrameGrabber::FrameGrabber(FrameQueue& q) :
    frameQueue_(q)
{
    stopEvent_.reset(CreateEvent(NULL, TRUE, FALSE, NULL));
}

//------------------------------------------------------------------------------
FrameGrabber::~FrameGrabber()
{
    stop();
}

//------------------------------------------------------------------------------
void FrameGrabber::start(const RECT&   rc,
                         HWND          hwnd,
                         const Format& fmt, 
                         HANDLE        hFrameAvailable, 
                         DWORD         startTimeMs)
{
    frameAvailableEvent_ = hFrameAvailable;
    framesPerSecondMax_          = fmt.getVideoFPS();
    framesPerSecond_             = framesPerSecondMax_ > 5 ? 5 : framesPerSecondMax_;

    recordedArea_ = rc;
    recordedWindow_ = hwnd;

    // set start time
    resumeTime_ = startOffset_ = startTimeMs;

    resume();
}

//------------------------------------------------------------------------------
void FrameGrabber::resume()
{
    // only resume if thread stopped
    if (WaitForSingleObject(thread_, 0) == WAIT_TIMEOUT)
        return;

    // set resume time
    startOffset_ = resumeTime_;

    // clear stop event
    ResetEvent(stopEvent_);

    unsigned threadId;
    HANDLE res = reinterpret_cast<HANDLE>(_beginthreadex(NULL, 0, threadProcStub, reinterpret_cast<LPVOID>(this), CREATE_SUSPENDED, &threadId));
    thread_.reset(res);
    ResumeThread(thread_);
}

//------------------------------------------------------------------------------
void FrameGrabber::stop()
{
    // don't stop if no thread's running
    if (WaitForSingleObject(thread_, 0) != WAIT_TIMEOUT)
        return;

    // tell the thread to exit
    SetEvent(stopEvent_);

    // wait until terminated
    WaitForSingleObject(thread_, INFINITE);
}

//------------------------------------------------------------------------------
int FrameGrabber::decreaseFps()
{
    if (framesPerSecond_ > 1) 
    {
        --framesPerSecond_;
    }

    return framesPerSecond_;
}

//------------------------------------------------------------------------------
int FrameGrabber::increaseFps()
{
    if (framesPerSecond_ < framesPerSecondMax_) 
    {
        ++framesPerSecond_;
    }

    return framesPerSecond_;
}

//------------------------------------------------------------------------------
unsigned int WINAPI FrameGrabber::threadProcStub(LPVOID lp)
{
    return reinterpret_cast<FrameGrabber*>(lp)->threadProc();
}

//------------------------------------------------------------------------------
unsigned int FrameGrabber::threadProc()
{
    HRESULT hr;

    ArgsInfo& args = ArgsInfo::getInstance();

    // create flashing frame window
    FrameWnd frame(recordedArea_, args.frameWidth_, args.frameLength_, args.frameColor1_, args.frameColor2_);

    // create screen capture object
    Grabber grabber(recordedArea_);

    // save start time
    startTime_ = timeGetTime();
    DWORD prevTime = (DWORD)-1;
    DWORD frameTime = (DWORD)-1;

    while (true)
    {
        // sleep a bit
        DWORD now = timeGetTime();
        DWORD ellapsed = now - prevTime;
        DWORD delta = 1000 / framesPerSecond_;
        DWORD remaining = (prevTime != -1 && ellapsed < delta) ? delta - ellapsed : 0;
        if (WaitForSingleObject(stopEvent_, remaining) != WAIT_TIMEOUT)
            break;

        prevTime = timeGetTime();

        // calculate frame time
        if (frameTime == (DWORD)-1) 
        {
            frameTime = startOffset_;
        }
        else 
        {
            frameTime = now - startTime_ + startOffset_;
        }

        // configure capture area
        if (recordedWindow_)
        {
            RECT rc;
            if (GetWindowRect(recordedWindow_, &rc) && !EqualRect(&rc, &recordedArea_)) 
            {
                OffsetRect(&recordedArea_, rc.left - recordedArea_.left, rc.top - recordedArea_.top);
                grabber.setScreenArea(recordedArea_);      
            }

            if (!isWindowVisible())
            {
                frame.hide();
                continue;
            }
        }

        // flash the frame
        frame.show();
        frame.flash(recordedArea_);

        // take a screen shot
        SetThreadPriority(thread_, THREAD_PRIORITY_BELOW_NORMAL);
        CComPtr<IBitMap> pIFrame;
        hr = grabber.captureScreen(&pIFrame);
        SetThreadPriority(thread_, THREAD_PRIORITY_NORMAL);

        if (FAILED(hr))
            continue;

        // append to queue
        frameQueue_.push(StampedFrame(frameTime, pIFrame));

        // signal availabilty of new frame
        SetEvent(frameAvailableEvent_);

        // check if we need to slow down...
        size_t framesLeft = frameQueue_.size();
        if (framesLeft > 10 || grabber.getPoolSize() > 20)
        {
            decreaseFps();
        }
        else if (framesLeft == 0 && grabber.getPoolSize() < 5)
        {
            increaseFps();
        }
    }

    resumeTime_ = timeGetTime() - startTime_ + startOffset_ + 1000 / framesPerSecond_;
    return 0;
}

//------------------------------------------------------------------------------
bool FrameGrabber::isWindowVisible() const
{
    if (!IsWindowVisible(recordedWindow_))
        return false;

    SmrtRgnHandle hRgn(CreateRectRgn(0,0,0,0));
    SmrtRgnHandle hRgnOther(CreateRectRgn(0,0,0,0));
    RECT rc;
    HWND hPrevWnd, hNextWnd;

    switch (GetWindowRgn(recordedWindow_, hRgn)) 
    {
    case NULLREGION:
        return false;

    case ERROR:
        if (!GetWindowRect(recordedWindow_, &rc))
            return false;
        SetRectRgn(hRgn, rc.left, rc.top, rc.right, rc.bottom);
        break;

    default:
        break;
    }

    for (hPrevWnd = recordedWindow_;
        (hNextWnd = GetWindow(hPrevWnd, GW_HWNDPREV)) != NULL;
        hPrevWnd = hNextWnd)
    {

        if (IsWindowVisible(hNextWnd)) {

            switch (GetWindowRgn(hNextWnd, hRgnOther)) 
            {
            case NULLREGION:
                continue;

            case ERROR:
                if (!GetWindowRect(hNextWnd, &rc))
                    continue;
                SetRectRgn(hRgnOther, rc.left, rc.top, rc.right, rc.bottom);
                break;

            default:
                break;
            }

            if (CombineRgn(hRgn, hRgn, hRgnOther, RGN_DIFF) == NULLREGION)
                return false;
        }

    }

    return true;
}

} // namespace camit
