//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "WaveRecorder.h"
#include <process.h>

namespace camit {

struct WaveFormats 
{
    DWORD   dwFormat;
    DWORD   nSamplesPerSec;
    WORD    nChannels;
    WORD    wBitsPerSample;
};

static WaveFormats waveFormats[] = 
{
    { WAVE_FORMAT_1M08, 11025, 1,  8 },
    { WAVE_FORMAT_1M16, 11025, 1, 16 },
    { WAVE_FORMAT_1S08, 11025, 2,  8 },
    { WAVE_FORMAT_1S16, 11025, 2, 16 },
    { WAVE_FORMAT_2M08, 22050, 1,  8 },
    { WAVE_FORMAT_2M16, 22050, 1, 16 },
    { WAVE_FORMAT_2S08, 22050, 2,  8 },
    { WAVE_FORMAT_2S16, 22050, 2, 16 },
    { WAVE_FORMAT_4M08, 44100, 1,  8 },
    { WAVE_FORMAT_4M16, 44100, 1, 16 },
    { WAVE_FORMAT_4S08, 44100, 2,  8 },
    { WAVE_FORMAT_4S16, 44100, 2, 16 }
};

//------------------------------------------------------------------------------
WaveRecorder::WaveRecorder(SoundQueue& queue) :
    sampleQueue_(queue)
{
    stopEvent_.reset(CreateEvent(NULL, TRUE, FALSE, NULL));
}

//------------------------------------------------------------------------------
WaveRecorder::~WaveRecorder()
{
}

//------------------------------------------------------------------------------
bool WaveRecorder::start(UINT           uDeviceID, 
                         const Format&  fmt, 
                         HANDLE         hBufferAvailable, 
                         DWORD          startTimeMs /* = 0 */)
{
    deviceId_             = uDeviceID;
    buffersPerSec_        = fmt.getAudioBuffersPerSec();
    bufferAvailableEvent_      = hBufferAvailable;

    // reset start time
    samplingTime_ = startTimeMs;

    return resume();
}

//------------------------------------------------------------------------------
bool WaveRecorder::resume()
{
    // only resume if thread stopped
    if (WaitForSingleObject(thread_, 0) == WAIT_TIMEOUT)
        return false;

    // clear stop event
    ResetEvent(stopEvent_);

    unsigned threadId;
    HANDLE res = reinterpret_cast<HANDLE>(_beginthreadex(NULL, 0, threadProcStub, reinterpret_cast<LPVOID>(this), CREATE_SUSPENDED, &threadId));
    thread_.reset(res);
    ResumeThread(thread_);

    return true;
}

//------------------------------------------------------------------------------
void WaveRecorder::stop()
{
    // don't stop if no thread's running
    if (WaitForSingleObject(thread_, 0) != WAIT_TIMEOUT)
        return;

    // stop thread
    SetEvent(stopEvent_);

    // wait for termination
    WaitForSingleObject(thread_, INFINITE);
}

//------------------------------------------------------------------------------
unsigned int WINAPI WaveRecorder::threadProcStub(LPVOID lp)
{
    WaveRecorder* pThis = reinterpret_cast<WaveRecorder*>(lp);
    return pThis->threadProc(lp);
}

//------------------------------------------------------------------------------
unsigned int WaveRecorder::threadProc(LPVOID lp)
{
    HANDLE hStopEvent = stopEvent_.get();

    // create a message queue for the thread
    MSG msg;
    PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE);

    // open the wave recording device
    if (waveInOpen(&inputDevice_, deviceId_, &waveFormat_, GetCurrentThreadId(), 0, CALLBACK_THREAD) != MMSYSERR_NOERROR) 
    {
        _tprintf(_T("Unable to open recording device\n"));
        return 1; 
    }

    // add a couple of buffers to the pipeline
    pendingBuffers_ = 0;
    for (int i = 0; i < 10; ++i) 
    {
        if (!addNewBuffer(inputDevice_)) return 1;
    }

    // start recording
    if (waveInStart(inputDevice_) != MMSYSERR_NOERROR) 
    {
        _tprintf(_T("Unable to start recording device\n"));
        return 1;
    }

    // re-add buffers by default
    bool bReAdd = true;

    // let's rock'n roll !
    while (pendingBuffers_ > 0) 
    {
        // wait until something happens
        if (MsgWaitForMultipleObjects(1, &hStopEvent, FALSE, INFINITE, QS_SENDMESSAGE|QS_POSTMESSAGE) == WAIT_OBJECT_0) 
        {
            // stop recording
            waveInReset(inputDevice_);
            bReAdd = false;
        }

        // process messages
        while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) 
        {
            switch (msg.message) 
            {
            case MM_WIM_OPEN:  break;
            case MM_WIM_CLOSE: break;
            case MM_WIM_DATA:

                if (bReAdd) 
                {
                    addNewBuffer(inputDevice_);
                }

                onMmWimData(inputDevice_, (WAVEHDR*)msg.lParam);
                break;

            default:  break;
            }
        }
    }

    // close recorder
    waveInClose(inputDevice_);
    return 0;
}

//------------------------------------------------------------------------------
void WaveRecorder::onMmWimData(HWAVEIN hInputDev, WAVEHDR* lpwvhdr)
{
    HRESULT hr;

    // extract the ISoundBuffer interface pointer from the header
    CComPtr<ISoundBuffer> pISound;
    hr = ISoundBuffer::GetInterfaceFromHeader(lpwvhdr, &pISound);
    if (FAILED(hr)) throw ComError(hr);

    // perform header clean-up 
    try 
    {
        pISound->UnprepareHeader(hInputDev);
        --pendingBuffers_;

        if (lpwvhdr->dwBytesRecorded > 0) 
        {

            // append to queue
            sampleQueue_.push(StampedSoundBuffer(samplingTime_, pISound));

            // signal availabilty
            SetEvent(bufferAvailableEvent_);
        }

        // update sample time
        samplingTime_ += 1000 * lpwvhdr->dwBytesRecorded / waveFormat_.nAvgBytesPerSec;

    }
    catch (...) 
    {
        // ignore the invalid buffer...
        _tprintf(_T("Invalid sound buffer\n"));
    }
}

//------------------------------------------------------------------------------
bool WaveRecorder::addNewBuffer(HWAVEIN hInputDev)
{
    HRESULT hr;

    CComPtr<ISoundBuffer> pISound(new SoundBuffer(waveFormat_.nAvgBytesPerSec / buffersPerSec_));

    // add to pipleline
    LPWAVEHDR pwhdr;
    hr = pISound->GetPreparedHeader(hInputDev, &pwhdr);
    if (FAILED(hr)) throw ComError(hr);

    if (waveInAddBuffer(hInputDev, pwhdr, sizeof(WAVEHDR)) != MMSYSERR_NOERROR)
        return false;

    ++pendingBuffers_;
    return true;
}

//------------------------------------------------------------------------------
bool WaveRecorder::selectFormat(UINT_PTR        uDeviceID, 
                                DWORD           nSamplesPerSec, 
                                WORD            wBitsPerSample, 
                                WORD            nChannels,
                                WAVEFORMATEX&   wfeRecorder)
{
    WAVEINCAPS wicaps;
    if (waveInGetDevCaps(uDeviceID, &wicaps, sizeof(WAVEINCAPS)) != MMSYSERR_NOERROR)
        return false;

    // check all formats supported by the device
    for (int i = 0; i < sizeof(waveFormats)/sizeof(WaveFormats); ++i) 
    {

        if (wicaps.dwFormats & waveFormats[i].dwFormat) 
        {

            // check if the format meets or exceeds the requested characterisics
            if (   waveFormats[i].nSamplesPerSec >= nSamplesPerSec
                && waveFormats[i].wBitsPerSample >= wBitsPerSample
                && waveFormats[i].nChannels >= nChannels)
            {
                // found a match
                waveFormat_.wFormatTag      = WAVE_FORMAT_PCM;
                waveFormat_.wBitsPerSample  = waveFormats[i].wBitsPerSample;
                waveFormat_.nSamplesPerSec  = waveFormats[i].nSamplesPerSec;
                waveFormat_.nChannels       = waveFormats[i].nChannels;
                waveFormat_.nBlockAlign     = waveFormat_.nChannels * waveFormat_.wBitsPerSample / 8;
                waveFormat_.nAvgBytesPerSec = waveFormat_.nSamplesPerSec * waveFormat_.nBlockAlign;
                waveFormat_.cbSize          = 0;

                // copy
                CopyMemory(&wfeRecorder, &waveFormat_, sizeof(WAVEFORMATEX) + waveFormat_.cbSize);
                return true;
            }
        }
    }

    // nothing found, reduce bits per sample
    if (wBitsPerSample == 16 && selectFormat(uDeviceID, nSamplesPerSec, 8, nChannels, wfeRecorder))
        return true;

    // nothing found, try mono
    if (nChannels == 2 && selectFormat(uDeviceID, nSamplesPerSec, wBitsPerSample, 1, wfeRecorder))
        return true;

    // nothing found, reduce bits per sample and switch to mono
    if (wBitsPerSample == 16 && nChannels == 2 && selectFormat(uDeviceID, nSamplesPerSec, 8, 1, wfeRecorder))
        return true;

    // lost hope...
    if (selectFormat(uDeviceID, 0, 8, 1, wfeRecorder))
        return true;

    return false;
}

//------------------------------------------------------------------------------
void WaveRecorder::dumpAudioDevices()
{
    unsigned devCnt = waveInGetNumDevs();

    if (devCnt > 0) 
    {

        _tprintf(_T("%2d : %s\n"), WAVE_MAPPER, _T("Default Recording Device"));

        for (unsigned i = 0; i < devCnt; ++i) 
        {
            WAVEINCAPS wicaps;
            if (waveInGetDevCaps(i, &wicaps, sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR
                && wicaps.dwFormats != 0) 
            {
                _tprintf(_T("%2d : %s\n"), i, wicaps.szPname);
            }
        }

    }
}

//------------------------------------------------------------------------------
void WaveRecorder::dumpRecordingFormats(UINT_PTR uDeviceID)
{
    WAVEINCAPS wicaps;
    if (waveInGetDevCaps(uDeviceID, &wicaps, sizeof(WAVEINCAPS)) == MMSYSERR_NOERROR) 
    {

        for (int i = 0; i < sizeof(waveFormats)/sizeof(WaveFormats); ++i) 
        {

            if (wicaps.dwFormats & waveFormats[i].dwFormat) 
            {
                _tprintf(_T("%2d : %6.3f kHz, %s %2d-bit\n"),  
                    i,
                    (double)waveFormats[i].nSamplesPerSec / 1000.0,
                    waveFormats[i].nChannels == 1 ? _T("mono,  ") : _T("stereo,"),
                    waveFormats[i].wBitsPerSample);
            }

        }
    }
}

} // namespace camit
