//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#ifndef CAMIT_WM_MEDIA_TYPE_H
#define CAMIT_WM_MEDIA_TYPE_H

namespace camit {

//------------------------------------------------------------------------------
// Class to encapsulate the WM_MEDIA_TYPE structure and provide reference
// counting.
//------------------------------------------------------------------------------
class WMMediaTypePtr
{
public:

    // constructor
    WMMediaTypePtr(WM_MEDIA_TYPE* pmt, bool bCopy = true) :
        mediaType_(NULL),
        refCount_(NULL)
    {
        try 
        {
            refCount_ = new long(1);

            if (pmt && bCopy) 
            {
                size_t size = sizeof(WM_MEDIA_TYPE) + pmt->cbFormat;
                BYTE* buf = new BYTE[size];
                memcpy(buf, pmt, size);
                mediaType_ = reinterpret_cast<WM_MEDIA_TYPE*>(buf);
            }
            else 
            {
                mediaType_ = pmt;
            }
        }
        catch (...) 
        {
            delete [] reinterpret_cast<BYTE*>(mediaType_);
            delete refCount_;
            throw;
        }
    }

    // construct by calling IWMVideoMediaProps::GetMediaType()
    WMMediaTypePtr(IWMMediaProps* pIMediaProps) :
        mediaType_(NULL),
        refCount_(NULL)
    {
        try 
        {
            HRESULT hr;

            refCount_ = new long(1);

            DWORD cnt = 0;
            hr = pIMediaProps->GetMediaType(NULL, &cnt);
            if (FAILED(hr)) throw ComError(hr);

            mediaType_ = (WM_MEDIA_TYPE*)new BYTE[cnt];
            hr = pIMediaProps->GetMediaType(mediaType_, &cnt);
            if (FAILED(hr)) throw ComError(hr);
        }
        catch (...) 
        {
            delete [] reinterpret_cast<BYTE*>(mediaType_);
            delete refCount_;
            throw;
        }
    }

    // copy constructor
    WMMediaTypePtr(const WMMediaTypePtr& rhs) :
        mediaType_(rhs.mediaType_),
        refCount_(rhs.refCount_)
    {
        ++(*refCount_);
    }

    // destructor
    ~WMMediaTypePtr()
    {
        destroy();
    }

    // assignment
    WMMediaTypePtr& operator=(const WMMediaTypePtr& rhs)
    {
        if (refCount_ != rhs.refCount_) 
        {
            ++(*rhs.refCount_);
            destroy();
            mediaType_ = rhs.mediaType_;
            refCount_ = rhs.refCount_;      
        }

        return (*this); 
    }

    // reset to new value (destroys previous)
    void reset(WM_MEDIA_TYPE* pmt)
    {
        if (--(*refCount_) == 0) 
        {
            delete [] reinterpret_cast<BYTE*>(mediaType_);
        }
        else 
        {
            try 
            {
                refCount_ = new long;
            }
            catch (...) 
            {
                delete [] reinterpret_cast<BYTE*>(mediaType_);
                throw;
            }
        }

        mediaType_ = pmt;
        *refCount_ = 1;
    }

    // release and return encapsulated pointer
    WM_MEDIA_TYPE* release() 
    { 
        if (*refCount_ > 1) 
        {
            long* ptmp = new long(1);
            --(*refCount_);
            refCount_ = ptmp;
        }

        WM_MEDIA_TYPE* tmp = mediaType_;
        mediaType_ = NULL;
        return tmp; 
    }

    // get the encapsulated pointer
    WM_MEDIA_TYPE* get() const
    { 
        return mediaType_;
    }

    // use WMMediaTypePtr just like a WM_MEDIA_TYPE
    operator WM_MEDIA_TYPE*() const 
    { 
        return mediaType_; 
    }

    // operator arrow
    WM_MEDIA_TYPE* operator->() const 
    { 
        return mediaType_; 
    }

    // dererference
    WM_MEDIA_TYPE& operator*() const 
    { 
        return *mediaType_; 
    }

    // comparison
    bool operator==(const WMMediaTypePtr& rhs) const { return mediaType_ == mediaType_; }
    bool operator!=(const WMMediaTypePtr& rhs) const { return mediaType_ != mediaType_; }

private:

    // destroy the encapsulated pointer
    void destroy() 
    { 
        if (--(*refCount_) == 0) 
        {
            delete [] reinterpret_cast<BYTE*>(mediaType_);
            delete refCount_;
        }
    }

    WM_MEDIA_TYPE*  mediaType_;
    long*           refCount_;
};


} // namespace camit

#endif // CAMIT_WM_MEDIA_TYPE_H


