//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "ArgsInfo.h"
#include "getopt.h"
#include "../hook/hook.h"
#include <stdio.h>
#include <mmsystem.h>
#include <Shellapi.h>
#include <commdlg.h>

#define PACKAGE _T("CamIt")
#define VERSION _T("1.6")

namespace camit {

// short option string (option letters followed by a colon ':' require an argument)
static const _TCHAR optstring[] = _T("hVvdAxo:O:t:a:n:D:w:l:r:s:p:e:g:T:U:X:f:B:q:C:u:NYSFW:H:c:b:k:iRzyI:K:m:J:L:");

// mapping of long to short arguments
static const getopt::Option longopts[] = {

// general options
{ _T("help"),                 no_argument,        NULL,   _T('h') },
{ _T("version"),              no_argument,        NULL,   _T('V') },
{ _T("verbose"),              no_argument,        NULL,   _T('v') },

// recording options
{ _T("output"),               required_argument,  NULL,   _T('o') },
{ _T("output-size"),          required_argument,  NULL,   _T('O') },
{ _T("disable-acceleration"), no_argument,        NULL,   _T('A') },
{ _T("save-dialog"),          no_argument,        NULL,   _T('x') },
{ _T("title"),                required_argument,  NULL,   _T('t') },
{ _T("author"),               required_argument,  NULL,   _T('a') },
{ _T("copyright"),            required_argument,  NULL,   _T('n') },
{ _T("description"),          required_argument,  NULL,   _T('D') },
{ _T("frame-width"),          required_argument,  NULL,   _T('w') },
{ _T("frame-length"),         required_argument,  NULL,   _T('l') },
{ _T("frame-color-1"),        required_argument,  NULL,   _T('r') },
{ _T("frame-color-2"),        required_argument,  NULL,   _T('s') },
{ _T("hilite-shape"),         required_argument,  NULL,   _T('p') },
{ _T("hilite-size"),          required_argument,  NULL,   _T('e') },
{ _T("hilite-color"),         required_argument,  NULL,   _T('g') },
{ _T("hotkey-stop"),          required_argument,  NULL,   _T('T') },
{ _T("hotkey-pause"),         required_argument,  NULL,   _T('U') },
{ _T("hotkey-resume"),        required_argument,  NULL,   _T('X') },

// video encoding options
{ _T("dump-video-codecs"),    no_argument,        NULL,   _T('d') },
{ _T("fps"),                  required_argument,  NULL,   _T('f') },
{ _T("color-depth"),          required_argument,  NULL,   _T('B') },
{ _T("video-quality"),        required_argument,  NULL,   _T('q') },
{ _T("cursor"),               required_argument,  NULL,   _T('C') },
{ _T("window-title"),         required_argument,  NULL,   _T('u') },
{ _T("window-select"),        no_argument,        NULL,   _T('N') },
{ _T("no-caption-buttons"),   no_argument,        NULL,   _T('Y') },
{ _T("select-area"),          no_argument,        NULL,   _T('S') },
{ _T("full-screen"),          no_argument,        NULL,   _T('F') },
{ _T("width"),                required_argument,  NULL,   _T('W') },
{ _T("height"),               required_argument,  NULL,   _T('H') },
{ _T("video-codec"),          required_argument,  NULL,   _T('c') },
{ _T("video-bitrate"),        required_argument,  NULL,   _T('b') },
{ _T("key-frame-period"),     required_argument,  NULL,   _T('k') },

// audio options
{ _T("dump-audio-devices"),   no_argument,        NULL,   _T('i') },
{ _T("show-rec-volumes"),     no_argument,        NULL,   _T('R') },
{ _T("show-rec-devices"),     no_argument,        NULL,   _T('z') },
{ _T("audio-disable"),        no_argument,        NULL,   _T('y') },
{ _T("audio-device"),         required_argument,  NULL,   _T('I') },
{ _T("audio-codec"),          required_argument,  NULL,   _T('K') },
{ _T("audio-sampling-rate"),  required_argument,  NULL,   _T('m') },
{ _T("audio-channels"),       required_argument,  NULL,   _T('J') },
{ _T("audio-resolution"),     required_argument,  NULL,   _T('L') },

{ NULL,                       no_argument,        NULL,   0,      }
};

//------------------------------------------------------------------------------
void ArgsInfo::usage() const
{
    _tprintf(_T("\nUsage: %s [OPTIONS]\n"), PACKAGE);

    // general options
    _tprintf(_T("\n  General options:\n\n"));
    _tprintf(_T("   -h            --help                    Print help and exit\n"                ));
    _tprintf(_T("   -V            --version                 Print version and exit\n"             ));
    _tprintf(_T("   -v            --verbose                 Give verbose feedback\n"              ));

    // recording options
    _tprintf(_T("\n  Recording options:\n\n"                                                      ));
    _tprintf(_T("   -o FILE       --output                   Output file name\n"                  ));
    _tprintf(_T("   -O FILE       --output-size              Output file name (& append size)\n"  ));
    _tprintf(_T("   -A            --disable-acceleration     Disable hw accel. during recording\n"));
    _tprintf(_T("   -x            --save-dialog              Display Save As dialog.\n"           ));
    _tprintf(_T("   -t TITLE      --title                    Clip title (default: none)\n"        ));
    _tprintf(_T("   -a AUTHOR     --author                   Clip author (default: none)\n"       ));
    _tprintf(_T("   -n COPYRIGHT  --copyright                Clip copyright (default: none)\n"    ));
    _tprintf(_T("   -D DESCRIP    --description              Clip description (default: none)\n"  ));
    _tprintf(_T("   -C FILE       --cursor                   Load custom cursor from .cur/.ico/\n")
            _T("                                            .ani file\n"                         )),
    _tprintf(_T("   -u TITLE      --window-title             Record only window with title\n"     )),
    _tprintf(_T("   -N            --window-select            Let user select a window\n"          )),
    _tprintf(_T("   -Y            --no-caption-buttons       Disable caption buttons\n"           )),
    _tprintf(_T("   -S            --select-area              Let user select capture area (def)\n")),
    _tprintf(_T("   -F            --full-screen              Grabber the entire screen\n"         )),
    _tprintf(_T("   -W WIDTH      --width                    Width of captured area\n"            )),
    _tprintf(_T("   -H HEIGHT     --height                   Height of captured area\n"           )),
    _tprintf(_T("   -w WIDTH      --frame-width              Width of flashing edges (def: 8)\n"  ));
    _tprintf(_T("   -l LENGTH     --frame-length             Length of flashing edges (def: 24)\n"));
    _tprintf(_T("   -r COLOR      --frame-color-1            Flashing color 1 (def: FFFFB4)\n"    ));
    _tprintf(_T("   -s COLOR      --frame-color-2            Flashing color 2 (def: 00FF50)\n"    ));
    _tprintf(_T("   -p SHAPE      --hilite-shape             Cursor hilite shape (Circle,\n"      )
            _T("                                            (Ellipse, Square, Rect, def: None)\n"));
    _tprintf(_T("   -e SIZE       --hilite-size              Cursor hilite shape size (def: 64)\n"));
    _tprintf(_T("   -g COLOR      --hilite-color             Cursor hilite color (def: FFFF7D)\n" ));
    _tprintf(_T("   -T KEY        --hotkey-stop              Hotkey for 'stop' (ex: SHIFT-F9)\n"  ));
    _tprintf(_T("   -U KEY        --hotkey-pause             Hotkey for 'pause' (ex: SHIFT-F9)\n" ));
    _tprintf(_T("   -X KEY        --hotkey-resume            Hotkey for 'resume' (ex: SHIFT-F9)\n"));

    // video encoding options
    _tprintf(_T("\n  Video options:\n\n"                                                          ));
    _tprintf(_T("   -d            --dump-video-codecs        List video codecs and exit\n"        ));
    _tprintf(_T("   -f FPS        --fps                      Max frames per second (def: 5)\n"    )),
    _tprintf(_T("   -B BITS       --color-depth              Color depth: 0(cur)/16(def)/24/32\n" ));
    _tprintf(_T("   -q QUALITY    --video-quality            Video quality for lossy codecs\n"    )),
    _tprintf(_T("   -c CODEC      --video-codec              Video codec (default: MSS1)\n"       ));
    _tprintf(_T("   -b BITRATE    --video-bitrate            Video bitrate kBit/sec (def: 200)\n" ));
    _tprintf(_T("   -k SECONDS    --key-frame-period         Sec's between key frames (def: 8)\n" ));

    // video encoding options
    _tprintf(_T("\n  Audio options:\n\n"));
    _tprintf(_T("   -i            --dump-audio-devices       List recording devices and exit\n"   ));
    _tprintf(_T("   -R            --show-rec-volumes         Show recording volume ctrl panel\n"  ));
    _tprintf(_T("   -z            --show-rec-devices         Show recording devices ctrl panel\n" ));
    _tprintf(_T("   -y            --audio-disable            Disable audio recording\n"           ));
    _tprintf(_T("   -I ID         --audio-device             Audio recording device\n"            ));
    _tprintf(_T("   -K CODEC      --audio-codec              Codec: WMAudio or ACELPnet (def)\n"  ));
    _tprintf(_T("   -m FREQ       --audio-sampling-rate      Sampling frequency (def: 16000 Hz)\n"));
    _tprintf(_T("   -J CHANNELS   --audio-channels           Channels: 1 (def) / 2 (stereo)\n"    ));
    _tprintf(_T("   -L BITS       --audio-resolution         Bits per sample: 8 or 16 (default)\n"));
}

//------------------------------------------------------------------------------
ArgsInfo& ArgsInfo::createInstance(int argc, _TCHAR* const* argv)
{
    ArgsInfo& argsInfo = getInstance();
    argsInfo.init(argc, argv);
    return argsInfo;
}

//------------------------------------------------------------------------------
ArgsInfo& ArgsInfo::getInstance()
{
    static ArgsInfo argsInfo;
    return argsInfo;
}

//------------------------------------------------------------------------------
ArgsInfo::ArgsInfo() :
    verbose_(false),
    dumpVideoCodecs_(false),
    dumpAudioDevices_(false),
    disableHardwareAcceleration_(false),
    appendSizeToFilename_(false),
    audioDeviceId_((UINT)-1),
    audioCodecGuid_(WMMEDIASUBTYPE_ACELPnet),
    audioSamplingFrequency_(16000),
    audioChannels_(1),
    audioResolution_(16),
    selectArea_(true),
    selectWindow_(false),
    noCaptionButtons_(false),
    fullScreen_(false),
    videoQuality_(70),
    captureAreaWidth_(-1),
    captureAreaHeight_(-1),
    frameWidth_(8),
    frameLength_(24),
    frameColor1_(RGB(255, 255, 180)),
    frameColor2_(RGB(0, 255, 80)),
    hiliteShape_(Grabber::CH_NONE),
    hiliteSize_(64),
    hiliteColor_(RGB(255, 255, 125)),
    hotkeyStop_(0),
    hotkeyPause_(0),
    hotkeyResume_(0),
    framesPerSec_(2),
    colorDepth_(16),
    fourCc_(MAKEFOURCC('M','S','S','1')),
    videoBitRate_(100 * 1024),
    keyFramePeriodSec_(8),
    recordedWindow_(NULL),
    clipTitle_(NULL),
    clipAuthor_(NULL),
    clipCopyright_(NULL)
{
  outputFile_[0] = _T('\0');
}

//------------------------------------------------------------------------------
void ArgsInfo::init(int argc, _TCHAR* const* argv)
{
    _tprintf(_T("%s %s, Copyright (C) 2002 - 2003  Daniel Gehriger\n"), PACKAGE, VERSION);
    _tprintf(_T("%s comes with ABSOLUTELY NO WARRANTY; for details\n"), PACKAGE);
    _tprintf(_T("see http://www.gnu.org/copyleft/gpl.html .\n\n"));

    getopt::optarg = 0;
    getopt::optind = 1;
    getopt::opterr = 1;
    getopt::optopt = '?';

    int requiredOptions = 0;
    bool bSaveAsDialog = false;

    while (true) 
    {
        int longIdx = 0;


        int c = getopt::getopt_long (argc, argv, optstring, longopts, &longIdx);

        if (c == -1) break; // Exit from `while (true)' loop.

        switch (c)
        {
        case _T('h'):       // Print help and exit.
            usage();
            exit (0);

        case _T('V'):       // Print version and exit.
            version();
            exit (0);

        case _T('v'):       // Be verbose
            verbose_ = true;
            break;

        case _T('d'):       // List available video codecs
            dumpVideoCodecs_ = true;
            break;

        case _T('i'):       // List available audio devices
            dumpAudioDevices_ = true;
            break;

        case _T('R'):       // Show volume control
            ShellExecute(NULL, _T("open"), _T("sndvol32.exe"), _T("/r"), NULL, SW_SHOW);
            exit(0);
            break;

        case _T('z'):       // Show volume control
            ShellExecute(NULL, _T("open"), _T("control.exe"), _T("MMSYS.CPL,,1"), NULL, SW_SHOW);
            exit(0);
            break;

        case _T('y'):       // Disable audio recording
            audioSamplingFrequency_ = 0;
            break;

        case _T('I'):       // Select the audio recording device
            audioDeviceId_ = _ttol(getopt::optarg);
            break;

        case _T('K'):       // Select audio codec
            if (_tcsnicmp(getopt::optarg, _T("WMAudio"), _tcslen(getopt::optarg)) == 0) 
            {
                audioCodecGuid_ = WMMEDIASUBTYPE_WMAudioV7;
            }
            else if (_tcsnicmp(getopt::optarg, _T("ACELPnet"), _tcslen(getopt::optarg)) == 0)
            {
                audioCodecGuid_ = WMMEDIASUBTYPE_ACELPnet;
            }
            else 
            {
                _ftprintf(stderr, _T("%s: invalid audio codec : %s."), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('m'):       // Set sampling frequency
            audioSamplingFrequency_ = _ttol(getopt::optarg);
            break;

        case _T('J'):       // Set audio channels
            audioChannels_ = static_cast<WORD>(_ttol(getopt::optarg));
            if (audioChannels_ != 1 && audioChannels_ != 2)
            {
                _ftprintf(stderr, _T("%s: invalid channel number: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('L'):       // Set audio resolution
            audioResolution_ = static_cast<WORD>(_ttol(getopt::optarg));
            if (audioResolution_ != 8 && audioResolution_ != 16) 
            {
                _ftprintf(stderr, _T("%s: invalid channel number: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('A'):      // set hardware acceleration
            disableHardwareAcceleration_ = true;
            break;

        case _T('x'): {    // display Save As dialog
            bSaveAsDialog = true;
            break; }

        case _T('o'):       // Specify output file
            _tcscpy(outputFile_, getopt::optarg);
            break;

        case _T('O'):       // Specify output file (with included size)
            _tcscpy(outputFile_, getopt::optarg);
            appendSizeToFilename_ = true;
            break;

        case _T('t'):       // Clip title
            clipTitle_ = getopt::optarg;
            break;

        case _T('a'):       // Clip author
            clipAuthor_ = getopt::optarg;
            break;

        case _T('n'):       // Clip copyright
            clipCopyright_ = getopt::optarg;
            break;

        case _T('D'):       // Clip copyright
            clipDescription_ = getopt::optarg;
            break;

        case _T('w'):       // Frame width
            frameWidth_ = _ttol(getopt::optarg);
            break;

        case _T('l'):       // Frame length
            frameLength_ = _ttol(getopt::optarg);
            break;

        case _T('r'): {     // Frame color 1
            unsigned r, g, b;
            TCHAR* optarg = getopt::optarg;
            if (*optarg == _T('#')) ++optarg;
            if (_stscanf(optarg, _T("%2x%2x%2x"), &r, &g, &b) != 3)
            {
                _ftprintf(stderr, _T("%s: invalid color specification: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }

            frameColor1_ = RGB(r, g, b);
            break; }

        case _T('s'): {     // Frame color 2
            unsigned r, g, b;
            TCHAR* optarg = getopt::optarg;
            if (*optarg == _T('#')) ++optarg;
            if (_stscanf(optarg, _T("%2x%2x%2x"), &r, &g, &b) != 3)
            {
                _ftprintf(stderr, _T("%s: invalid color specification: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }

            frameColor2_ = RGB(r, g, b);
            break; }

        case _T('p'): {     // hilite shape
            static const TCHAR* shapeNames[5] = { _T("circle"), _T("ellipse"), _T("square"), _T("rectangle"), _T("none") };
            static const Grabber::Shape shapes[5] = { Grabber::CH_CIRCLE, Grabber::CH_ELLIPSE, Grabber::CH_SQUARE, Grabber::CH_RECTANGLE, Grabber::CH_NONE };
            int i;
            for (i = 0; i < 5; ++i)
            {
                if (_tcsnicmp(getopt::optarg, shapeNames[i], _tcslen(getopt::optarg)) == 0)
                    break;
            }
            if (i == 5) 
            {
                _ftprintf(stderr, _T("%s: invalid shape name: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }
            hiliteShape_ = shapes[i];
            break; }

        case _T('e'):       // hilite size
            hiliteSize_ = _ttol(getopt::optarg);
            if (hiliteSize_ < 1) 
            {
                _ftprintf(stderr, _T("%s: invalid hilite size: %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('g'): {     // hilite color
            unsigned r, g, b;
            TCHAR* optarg = getopt::optarg;
            if (*optarg == _T('#')) ++optarg;
            if (_stscanf(optarg, _T("%2x%2x%2x"), &r, &g, &b) != 3)
            {
                _ftprintf(stderr, _T("%s: invalid color specification: %s."), PACKAGE, getopt::optarg);
                exit(1);
            }

            hiliteColor_ = RGB(r, g, b);
            break; }

        case _T('T'):       // hotkey 'stop'
            hotkeyStop_ = parseHotKey(getopt::optarg);
            if (hotkeyStop_ == 0) 
            {
                _ftprintf(stderr, _T("%s: invalid hotkey : %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('U'):       // hotkey 'pause'
            hotkeyPause_ = parseHotKey(getopt::optarg);
            if (hotkeyStop_ == 0)
            {
                _ftprintf(stderr, _T("%s: invalid hotkey : %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('X'):       // hotkey 'resume'
            hotkeyResume_ = parseHotKey(getopt::optarg);
            if (hotkeyStop_ == 0)
            {
                _ftprintf(stderr, _T("%s: invalid hotkey : %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('u'):       // Record specified window
            windowTitle_ = getopt::optarg;

            if (recordedWindow_ = FindWindow(NULL, windowTitle_))
                break;

            EnumWindows(enumWindowsProc, reinterpret_cast<LPARAM>(this));
            if (recordedWindow_ == NULL) 
            {
                _ftprintf(stderr, _T("%s: cannot find window : %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('N'):       // User selects a window
            selectWindow_ = true;
            selectArea_   = false;
            break;

        case _T('Y'):       // Disable caption buttons
            noCaptionButtons_ = true;
            break;

        case _T('S'):       // Select screen area
            selectArea_ = true;
            fullScreen_ = false;
            captureAreaWidth_ = -1;
            captureAreaHeight_ = -1;
            break;

        case _T('F'):       // Grabber full screen
            fullScreen_ = true;
            selectArea_ = false;
            captureAreaWidth_ = -1;
            captureAreaHeight_ = -1;
            break;

        case _T('W'):       // Specify captured width
            captureAreaWidth_ = _ttol(getopt::optarg);
            fullScreen_ = false;
            selectArea_ = false;
            break;

        case _T('H'):       // Specify captured height
            captureAreaHeight_ = _ttol(getopt::optarg);
            fullScreen_ = false;
            selectArea_ = false;
            break;


        case _T('f'):       // FPS
            framesPerSec_ = _ttol(getopt::optarg);
            if (framesPerSec_ < 1)
            {
                _ftprintf(stderr, _T("%s: invalid frames per second: %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case  _T('B'):      // force color depth
            colorDepth_ = static_cast<WORD>(_ttol(getopt::optarg));
            switch (colorDepth_) {
        case 0:
        case 16:
        case 24:
        case 32:
            break;
        default:
            _ftprintf(stderr, _T("%s: invalid color depth (valid values are 0, 16, 24, 32): %s.\n"), PACKAGE, getopt::optarg);
            exit(1);
            }
            break;

        case _T('q'):       // video quality
            videoQuality_ = _ttol(getopt::optarg);
            if (videoQuality_ < 0 || videoQuality_ > 100) 
            {
                _ftprintf(stderr, _T("%s: invalid video quality (must be 0-100): %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('C'):       // load custom cursor
            customCursor_ = LoadCursorFromFile(getopt::optarg);
            if (!customCursor_) 
            {
                _ftprintf(stderr, _T("%s: invalid cursor file: %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case _T('c'):       // Specify video codec
            if (_tcslen(getopt::optarg) != 4) 
            {
                _ftprintf(stderr, _T("%s: invalid codec: %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
#ifdef _UNICODE
            char codec[5];
            if (WideCharToMultiByte(CP_ACP, 0, getopt::optarg, 4, codec, 5, NULL, NULL))
            {
                fourCc_ = MAKEFOURCC(toupper(codec[0]), 
                                     toupper(codec[1]), 
                                     toupper(codec[2]), 
                                     toupper(codec[3]));
            }
#else
            fourCc_ = MAKEFOURCC(toupper(getopt::optarg[0]), 
                                 toupper(getopt::optarg[1]), 
                                 toupper(getopt::optarg[2]), 
                                 toupper(getopt::optarg[3]));
#endif
            break;

        case _T('b'):
            videoBitRate_ = _ttol(getopt::optarg) * 1024;
            break;

        case _T('k'):
            keyFramePeriodSec_ = _ttol(getopt::optarg);
            if (keyFramePeriodSec_ < 1) 
            {
                _ftprintf(stderr, _T("%s: invalid key frame period: %s.\n"), PACKAGE, getopt::optarg);
                exit(1);
            }
            break;

        case 0:             // Long option with no short option
            break;

        case _T('?'):       // Invalid option.
            // `getopt_long' already printed an error message.
            exit (1);

        default:            // Option not considered.
            _ftprintf (stderr, _T("%s: option unknown: %c\n"), PACKAGE, c);
            exit(1);
        }
    }

    if (requiredOptions)
        exit(1);

    if (bSaveAsDialog)
    {
        OPENFILENAME ofn;
        ZeroMemory(&ofn, sizeof(ofn));
        ofn.lStructSize = sizeof(ofn);
        ofn.lpstrFilter = _T("Windows Media file (*.wmv)\0*.wmf");
        ofn.nFilterIndex= 1;
        ofn.lpstrDefExt = _T("wmv");
        ofn.lpstrFile   = outputFile_;
        ofn.nMaxFile    = MAX_PATH;
        ofn.lpstrTitle  = _T("Save Recording As");
        ofn.Flags       = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
        if (!GetSaveFileName(&ofn)) 
        {
            exit(0);
        }
    }
}

//------------------------------------------------------------------------------
void ArgsInfo::version() const
{
    _tprintf(_T("%s %s\n"), PACKAGE, VERSION);
}

//------------------------------------------------------------------------------
DWORD ArgsInfo::parseHotKey(TCHAR* str) const
{
    WORD vkHotkey = 0;
    WORD modifier = 0;
    TCHAR* psz = str;

    while (*psz) 
    {

        if (*psz == _T('-') || *psz == _T('+')) 
        {
            ++psz;
            continue;
        }

        if (_tcsnicmp(psz, _T("CTRL"), 4) == 0)
        {
            modifier |= MOD_CONTROL;
            psz += 4;
            continue;
        }

        if (_tcsnicmp(psz, _T("CTL"), 3) == 0) 
        {
            modifier |= MOD_CONTROL;
            psz += 3;
            continue;
        }

        if (_tcsnicmp(psz, _T("SHIFT"), 5) == 0) 
        {
            modifier |= MOD_SHIFT;
            psz += 5;
            continue;
        }

        if (_tcsnicmp(psz, _T("ALT"), 5) == 0) 
        {
            modifier |= MOD_ALT;
            psz += 3;
            continue;
        }

        if (_tcsnicmp(psz, _T("WIN"), 5) == 0) 
        {
            modifier |= MOD_WIN;
            psz += 3;
            continue;
        }

        // Fx
        if (tolower(*psz) == _T('f') && *(psz+1) && _istdigit(*(psz+1)))
            return MAKELONG(modifier, _ttol(psz+1) + VK_F1 - 1);

        // regular character
        if (_istalpha(*psz) && *(psz + 1) == 0)
            return MAKELONG(modifier, toupper(*psz));

        return 0;
    }

    return 0;
}

//------------------------------------------------------------------------------
BOOL CALLBACK ArgsInfo::enumWindowsProc(HWND hwnd, LPARAM lParam)
{
    ArgsInfo* pThis = reinterpret_cast<ArgsInfo*>(lParam);

    if (HWND hwndChild = FindWindowEx(hwnd, NULL, NULL, pThis->windowTitle_)) 
    {
        pThis->recordedWindow_ = hwndChild;
        return FALSE;
    }

    return TRUE;
}

} // namespace camit
