//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "BitMap.h"
#include <WindowsX.h>

namespace camit {

// {9A3DA201-354E-4b75-AB20-D9B748269444}
const IID IID_IBitMap = { 0x9a3da201, 0x354e, 0x4b75, { 0xab, 0x20, 0xd9, 0xb7, 0x48, 0x26, 0x94, 0x44 } };

//------------------------------------------------------------------------------
BitMap::BitMap(LONG width, LONG height, WORD bits)
{
    // DWORD align the width of the DIB
    unsigned lineLength  = (width * bits + 31) / 32 * 4;

    // figure out the size of the colour table
    size_t colorTableSize = sizeof(RGBQUAD) * ((bits <= 8) ? 1 << bits : 0);

    // calculate the size of the DIB
    size_t size = sizeof(BITMAPINFOHEADER) + colorTableSize + lineLength * height;

    // allocate buffer (if the requested block is larger than 1 Meg, we bypass
    // the heap manager.
    LPVOID buf = VirtualAlloc(NULL, size, MEM_RESERVE|MEM_COMMIT|MEM_TOP_DOWN, PAGE_READWRITE);
    if (buf == NULL) throw ComError(E_OUTOFMEMORY);

    // initialize BITMAPINFOHEADER
    bmih_ = reinterpret_cast<LPBITMAPINFOHEADER>(buf);

    bmih_->biSize          = sizeof(BITMAPINFOHEADER);
    bmih_->biWidth         = width;
    bmih_->biHeight        = height;
    bmih_->biPlanes        = 1;
    bmih_->biBitCount      = bits;
    bmih_->biCompression   = BI_RGB;
    bmih_->biSizeImage     = (DWORD)(size - sizeof(BITMAPINFOHEADER) - colorTableSize);
    bmih_->biXPelsPerMeter = 0;
    bmih_->biYPelsPerMeter = 0;
    bmih_->biClrUsed       = (bits <= 8) ? 1 << bits : 0;
    bmih_->biClrImportant  = 0;
}

//------------------------------------------------------------------------------
BitMap::~BitMap()
{
    // figure out the size of the colour table
    size_t colorTableSize = sizeof(RGBQUAD) * ((bmih_->biBitCount <= 8) ? 1 << bmih_->biBitCount : 0);

    // calculate size of allocated block
    size_t size = bmih_->biSizeImage + sizeof(BITMAPINFOHEADER) + colorTableSize;

    VirtualFree(bmih_, 0, MEM_RELEASE);
}

//------------------------------------------------------------------------------
// Override this to say what interfaces we support and where
//------------------------------------------------------------------------------
STDMETHODIMP BitMap::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
    if (riid == IID_INSSBuffer) 
    {
        *ppv = static_cast<INSSBuffer*>(this);
        static_cast<INSSBuffer*>(this)->AddRef();
        return S_OK;
    }
    else if (riid == IID_IBitMap)
    {
        *ppv = static_cast<IBitMap*>(this);
        static_cast<IBitMap*>(this)->AddRef();
        return S_OK;
    } 
    else 
    {
        return IUnknownImpl::NonDelegatingQueryInterface(riid, ppv);
    }
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::GetBitMapInfoHeader(LPBITMAPINFOHEADER* ppBmi)
{
    if (ppBmi == NULL) return E_POINTER;
    *ppBmi = bmih_;
    return S_OK;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::ReUse(LONG width, LONG height, WORD bits)
{
    if (refCount_ == 1) 
    {
        if (   bmih_->biWidth == width
            && bmih_->biHeight == height
            && bmih_->biBitCount == bits)
        {
            return S_OK;
        }
        else
        {
            return E_INVALIDARG;
        }
    }
    else 
        return E_FAIL;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::GetLength(DWORD* pdwLength)
{
    if (pdwLength == NULL) return E_POINTER;
    *pdwLength = DIBSIZE(*bmih_);
    return S_OK;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::SetLength(DWORD dwLength)
{
    return E_NOTIMPL;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::GetMaxLength(DWORD* pdwLength)
{
    if (pdwLength == NULL) return E_POINTER;
    *pdwLength = DIBSIZE(*bmih_);
    return S_OK;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::GetBuffer(BYTE** ppdwBuffer)
{
    if (ppdwBuffer == NULL) return E_POINTER;
    *ppdwBuffer = (LPBYTE)bmih_ + bmih_->biSize + bmih_->biClrUsed * sizeof(RGBQUAD);
    return S_OK;
}

//------------------------------------------------------------------------------
STDMETHODIMP BitMap::GetBufferAndLength(BYTE** ppdwBuffer, DWORD* pdwLength)
{
    if (ppdwBuffer == NULL || pdwLength == NULL) return E_POINTER;
    *pdwLength = DIBSIZE(*bmih_);
    *ppdwBuffer = (LPBYTE)bmih_ + bmih_->biSize + bmih_->biClrUsed * sizeof(RGBQUAD);
    return S_OK;
}

} // namespace camit
