//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "CamIt.h"
#include "SelWnd.h"
#include "util.h"
#include "SmrtHandle.h"
#include "WMProfiles.h"
#include "Grabber.h"
#include "WindowFinder.h"
#include "../hook/hook.h"
#include "../res/resource.h"

namespace camit {

    // number of frames and sound buffers to keep in pipeline
#define MIN_PIPELINE_ENTRIES 2

    // max number of entries before processing
#define MAX_PIPELINE_ENTRIES 8

//------------------------------------------------------------------------------
CamIt::CamIt() :
    frameGrabber_(frameQueue_),
    waveRecorder_(soundQueue_),
    recordedWindow_(NULL),
    titleBarWindow_(NULL)
{
    // determine screen dimensions
    SmrtDcRelease hDc(NULL, GetDC(NULL));
    desktopWidth_ = GetDeviceCaps(hDc, HORZRES);
    desktopHeight_ = GetDeviceCaps(hDc, VERTRES);

    // init recording rect
    SetRectEmpty(&recordedArea_);

    // install hook
    installCurHook();
}

//------------------------------------------------------------------------------
CamIt::~CamIt()
{
    uninstallCurHook();
}

//------------------------------------------------------------------------------
bool CamIt::record(const ArgsInfo& args)
{
    HRESULT hr;

    // configure output format
    Format fmt;
    fmt.setClipAttributes(args.clipTitle_, args.clipAuthor_, args.clipCopyright_, args.clipDescription_);
    fmt.setVideoCodec(args.fourCc_);
    fmt.setVideoFPS(args.framesPerSec_);
    fmt.setVideoBitRate(args.videoBitRate_);
    fmt.setVideoQuality(args.videoQuality_);
    fmt.setVideoSecPerKey(args.keyFramePeriodSec_);

    fmt.setAudioCodec(args.audioCodecGuid_);
    fmt.setAudioSamplingRate(args.audioSamplingFrequency_);
    fmt.setAudioChannels(args.audioChannels_);
    fmt.setAudioResolution(args.audioResolution_);

    // configure audio recording device
    if (args.audioSamplingFrequency_ == 0) 
    {
        soundEnabled_ = false;
    }
    else 
    {
        soundEnabled_ = true;
        if (!waveRecorder_.selectFormat(args.audioDeviceId_,
                                         args.audioSamplingFrequency_,
                                         args.audioResolution_,
                                         args.audioChannels_,
                                         fmt.waveFormatRecorder_))
        {
            soundEnabled_ = false;
        }
    }

    int i;
    for (i = 0; i < 4; ++i) 
    {
        CComPtr<IBitMap> pIBitMap;
        hr = Grabber(recordedArea_).captureScreen(&pIBitMap);
        if (FAILED(hr)) throw ComError(hr);

        LPBITMAPINFOHEADER pbih;
        pIBitMap->GetBitMapInfoHeader(&pbih);
        fmt.setBitMapInfoHeader(pbih);

        if (wmFile_.testAlignment(fmt))
            break;

        alignRect(1 << (i+1), desktopWidth_, desktopHeight_, recordedArea_);
    }
    if (i == 4)
        return false;

    // create a message queue for the thread
    MSG msg;
    PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE);

    // register the commands & hotkeys
    registerHotKeys(args);

    // create output file
    TCHAR path[_MAX_PATH];
    _tcscpy(path, args.outputFile_);
    TCHAR* dot = _tcsrchr(path, _T('.'));
    if (!dot) { dot = path + _tcslen(path); }
    if (args.appendSizeToFilename_) 
    {
        _stprintf(dot, _T("[%dx%d].wmv"), recordedArea_.right - recordedArea_.left, recordedArea_.bottom - recordedArea_.top);
    }
    else 
    {
        _tcscpy(dot, _T(".wmv"));
    }
    wmFile_.open(path, fmt, !soundEnabled_);

    // create waitable events
    SmrtWin32Handle hFramesAvailable(CreateEvent(NULL, FALSE, FALSE, NULL));
    SmrtWin32Handle hSoundAvailable(CreateEvent(NULL, FALSE, FALSE, NULL));
    HANDLE handles[] = { hFramesAvailable.get(), hSoundAvailable.get() };

    // start frame grabber
    frameGrabber_.start(recordedArea_, recordedWindow_, fmt, hFramesAvailable);

    // start sound recorder
    if (soundEnabled_) 
    {
        if (!waveRecorder_.start(args.audioDeviceId_, fmt, hSoundAvailable)) 
        {
            MessageBox(NULL, _T("Unable to start audio recording device"), _T("Recording Error"), MB_OK | MB_ICONWARNING);
            return false;
        }
    }

    // start console thread
    console_.start();


    // and let's go !
    while (writesSamples(false)) 
    {

        // wait for new frames / sounds / hotkeys
        MsgWaitForMultipleObjects(2, handles, FALSE, INFINITE, QS_ALLEVENTS);

    }

    frameGrabber_.stop();
    if (soundEnabled_) { waveRecorder_.stop(); }
    console_.stop();

    // write outstanding frames
    writesSamples(true);

    wmFile_.close();
    return true;
}

// -----------------------------------------------------------------------------
bool CamIt::writesSamples(bool bFinal)
{
    HRESULT hr;

    bool bPaused = false;
    bool bStopped = false;

    // minimum of sample/frames to keep in queue
    size_t minSize = bFinal ? 0 : MIN_PIPELINE_ENTRIES;
    size_t maxSize = MAX_PIPELINE_ENTRIES;

    // send all frames to encoder
    while (true) 
    {

        // handle hotkeys / commands
        MSG msg;
        while (!bFinal && PeekMessage(&msg, NULL, WM_HOTKEY, WM_HOTKEY, PM_REMOVE)) 
        {

            if (msg.message == WM_HOTKEY) 
            {

                UINT id = static_cast<UINT>(msg.wParam);

                if (msgIdPause_.equals(id)) 
                {
                    // pause recording
                    frameGrabber_.stop();
                    if (soundEnabled_) { waveRecorder_.stop(); }
                    bPaused = true;

                    // install "resume" button
                    if (titleBarWindow_) 
                    {
                        PostMessage(titleBarWindow_, windowMsgResume_, (WPARAM)msgIdResume_, (LPARAM)0);
                    }
                }
                else if (msgIdResume_.equals(id)) 
                {
                    // resume recording
                    frameGrabber_.resume();
                    if (soundEnabled_) { waveRecorder_.resume(); }

                    // re-install "pause" button
                    if (titleBarWindow_) 
                    {
                        PostMessage(titleBarWindow_, windowMsgPause_, (WPARAM)msgIdPause_, (LPARAM)0);
                    }
                }
                else if (msgIdStop_.equals(id)) 
                {
                    // stop recording
                    frameGrabber_.stop();
                    if (soundEnabled_) { waveRecorder_.stop(); }
                    bStopped = true;
                }
            }

        }

        // find next sample or frame
        size_t frameCnt = frameQueue_.size();
        size_t sampleCnt = soundQueue_.size();

        // if we are recording both sound & video, always keep the queues filled
        // with a few samples/frames in order to make sure that we don't write a 
        // sample/frame with a time stamp later than the time stamp of one that 
        // is not yet in the queue.
        if (soundEnabled_ && (frameCnt < minSize || sampleCnt < minSize ) && frameCnt < maxSize && sampleCnt < maxSize)
            break;

        if (frameCnt == 0 && sampleCnt == 0)
            break;

        StampedFrame* frame = frameCnt > 0 ? &frameQueue_.front() : NULL;
        StampedSoundBuffer* sound = sampleCnt > 0 ? &soundQueue_.front() : NULL;

        // write sound
        if (frame && (!sound || frame->first < sound->first)) 
        {
            // obtain INSSBuffer interface of captured frame
            CComPtr<INSSBuffer> pIBuffer;
            hr = frame->second->QueryInterface(IID_INSSBuffer, (void**)&pIBuffer);
            if (FAILED(hr)) throw ComError(hr);

            wmFile_.writeVideoSample(pIBuffer, frame->first);
            frameQueue_.pop();
        }
        else if (sound) 
        {
            // obtain INSSBuffer interface of captured sample
            CComPtr<INSSBuffer> pIBuffer;
            hr = sound->second->QueryInterface(IID_INSSBuffer, (void**)&pIBuffer);
            if (FAILED(hr)) throw ComError(hr);

            wmFile_.writeAudioSample(pIBuffer, sound->first);
            soundQueue_.pop();
        }

    }

    if (bPaused) 
    {
        wmFile_.key();
        wmFile_.discontinuity();
    }

    return !bStopped;
}

//------------------------------------------------------------------------------
void CamIt::registerHotKeys(const ArgsInfo& args)
{
    // unique atom names (names don't matter, but must be unique throughout the system)
    static const TCHAR* szAtomNamePause  =  _T("60B8061D-72E7-11D6-A763-00C0F04895FC");
    static const TCHAR* szAtomNameStop   =  _T("532FFA9E-72E7-11D6-A763-00C0F04895FC");
    static const TCHAR* szAtomNameResume =  _T("60B8061F-72E7-11D6-A763-00C0F04895FC");

    // the command names (these *do* matter; an exclamation mark '!' will be appended
    // automatically)
    static const char* szCommandPause  = "PAUSE";
    static const char* szCommandStop   = "STOP";
    static const char* szCommandResume = "RESUME";

    // create unique hotkey identifiers
    msgIdPause_.reset(GlobalAddAtom(szAtomNamePause));
    msgIdStop_.reset(GlobalAddAtom(szAtomNameStop));
    msgIdResume_.reset(GlobalAddAtom(szAtomNameResume));

    // register hotkeys with console
    console_.registerCommand(szCommandPause, msgIdPause_, args.hotkeyPause_);
    console_.registerCommand(szCommandStop, msgIdStop_, args.hotkeyStop_);
    console_.registerCommand(szCommandResume, msgIdResume_, args.hotkeyResume_);

    // register hotkeys with system
    if (args.hotkeyPause_ && RegisterHotKey(NULL, msgIdPause_, LOWORD(args.hotkeyPause_), HIWORD(args.hotkeyPause_))) 
    {
        hotkeyPause_.reset(NULL, (int)msgIdPause_.get());
    }

    if (args.hotkeyStop_ && RegisterHotKey(NULL, msgIdStop_, LOWORD(args.hotkeyStop_), HIWORD(args.hotkeyStop_))) 
    {
        hotkeyStop_.reset(NULL, (int)msgIdStop_.get());
    }

    if (args.hotkeyResume_ && RegisterHotKey(NULL, msgIdResume_, LOWORD(args.hotkeyResume_), HIWORD(args.hotkeyResume_))) 
    {
        hotkeyResume_.reset(NULL, (int)msgIdResume_.get());
    }

    // add window buttons
    if (!args.noCaptionButtons_ && recordedWindow_) 
    {
        // find top level window
        titleBarWindow_ = recordedWindow_;
        if (!(static_cast<DWORD>(GetWindowLong(titleBarWindow_, GWL_STYLE)) & WS_CAPTION)) 
        {
            EnumWindows(enumWindowsProc, reinterpret_cast<LPARAM>(this));
        }

        if (static_cast<DWORD>(GetWindowLong(titleBarWindow_, GWL_STYLE)) & WS_CAPTION) 
        {
            windowMsgStop_ = RegisterWindowMessage(WM_USER_INSTALL_STOP_BUTTON_NAME);
            windowMsgPause_ = RegisterWindowMessage(WM_USER_INSTALL_PAUSE_BUTTON_NAME);
            windowMsgResume_ = RegisterWindowMessage(WM_USER_INSTALL_RESUME_BUTTON_NAME);
            PostMessage(titleBarWindow_, windowMsgStop_, (WPARAM)msgIdStop_, (LPARAM)args.hotkeyStop_);
            PostMessage(titleBarWindow_, windowMsgPause_, (WPARAM)msgIdPause_, (LPARAM)args.hotkeyPause_);
        }
    }
}

//------------------------------------------------------------------------------
BOOL CALLBACK CamIt::enumWindowsProc(HWND hwndParent, LPARAM lParam) 
{
    CamIt* pThis = reinterpret_cast<CamIt*>(lParam);

    if (IsChild(hwndParent, pThis->recordedWindow_)) 
    {
        pThis->titleBarWindow_ = hwndParent;
        return FALSE;
    }

    return TRUE;
}

//------------------------------------------------------------------------------
void CamIt::setRecordedWindow(HWND hwnd)
{
    recordedWindow_ = hwnd;
    GetWindowRect(hwnd, &recordedArea_);
}

//------------------------------------------------------------------------------
void CamIt::setFullScreen()
{
    SetRect(&recordedArea_, 0, 0, desktopWidth_, desktopHeight_);
}

//------------------------------------------------------------------------------
void CamIt::chooseRecordedArea(long width /* = -1 */, long height /* = -1 */)
{
    if (width > 0 && height > 0) 
    {
        SelWnd wnd(SelWnd::RM_FIXED, width, height);
        wnd.run();
        recordedArea_ = wnd.getRegion();
    }
    else 
    {
        SelWnd wnd(SelWnd::RM_VARIABLE);
        wnd.run();
        recordedArea_ = wnd.getRegion();
    }
}

//------------------------------------------------------------------------------
bool CamIt::chooseRecordedWindow()
{
    WindowFinder fnd;
    HWND hwnd = fnd.startsearchWindowDialog(NULL);
    if (hwnd == NULL)
        return false;

    setRecordedWindow(hwnd);
    return true;
}

} // namespace camit



