//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "Grabber.h"
#include "ArgsInfo.h"
#include "../hook/hook.h"
#include <stdlib.h>
#include <iterator>

#define SAFE_RELEASE(p) { if(p) { (p)->Release(); (p)=NULL; } }

namespace camit {

// -----------------------------------------------------------------------------
Grabber::Grabber(const RECT& rc) :
    captureCursor_(true),
    spareBitmaps_(1),
    width_(0),
    height_(0)
{
    // retrieve arguments
    ArgsInfo& args = ArgsInfo::getInstance();
    hiliteShape_ = args.hiliteShape_;
    hiliteColor_ = args.hiliteColor_;
    hiliteSize_  = args.hiliteSize_;
    colorDepth_ = args.colorDepth_;

    // check if we should use a custom cursor
    customCursor_ = args.customCursor_;
    if (!customCursor_) 
    {

        // install cursor hook
        installCurHook();

        // load default cursor
        defaultCursor_ = LoadCursor(NULL, IDC_ARROW);
    }

    // get handle to desktop
    desktopDc_.reset(NULL, GetDC(NULL));

    // prepare a compatible memory device
    memoryDc_.reset(CreateCompatibleDC(desktopDc_));

    if (!IsRectEmpty(&rc)) 
    {
        setScreenArea(rc);
    }
}

// -----------------------------------------------------------------------------
Grabber::~Grabber()
{
    // release pooled bitmaps
    for (BitMapPool::iterator it = bitMaps_.begin(); it != bitMaps_.end(); ++it) 
    {
        (*it)->GetUnknown()->Release();
    }

    // remove cursor hook (the hook is reference counted)
    uninstallCurHook();
}

//------------------------------------------------------------------------------
void Grabber::setScreenArea(const RECT& rc)
{
    // set region
    long left    = rc.left;
    long top     = rc.top;
    long width   = rc.right - rc.left;
    long height  = rc.bottom - rc.top;

    if (width != width_ || height != height_) 
    {
        // prepare a compatible bm
        bitmapDc_.reset(CreateCompatibleBitmap(desktopDc_, width, height));
    }

    left_ = left;
    top_ = top;
    width_ = width;
    height_ = height;
}

// -----------------------------------------------------------------------------
HRESULT Grabber::captureScreen(IBitMap** ppIBitMap) const
{
    // select the bm into the memory device
    SmrtGdiSelect oldbm(memoryDc_, SelectObject(memoryDc_, bitmapDc_));

    // and copy !
    if (!BitBlt(memoryDc_, 0, 0, width_, height_, desktopDc_, left_, top_, SRCCOPY))
        return CAMIT_HRESULT_FROM_WIN32(GetLastError());

    // Get Cursor Pos
    POINT pt;
    GetCursorPos(&pt);

    HCURSOR hcur = fetchCursorHandle();
    pt.x -= left_;
    pt.y -= top_;

    // Draw the HighLight
    if (hiliteShape_ != CH_NONE) 
    {
        POINT ptHilite = { pt.x - 64, pt.y - 64 };
        hiliteCursor(memoryDc_, ptHilite.x, ptHilite.y);
    }

    // Draw the Cursor
    if (captureCursor_) 
    {
        ICONINFO  iconinfo;
        if (GetIconInfo(hcur, &iconinfo)) 
        {
            pt.x -= iconinfo.xHotspot;
            pt.y -= iconinfo.yHotspot;
            if (iconinfo.hbmMask) DeleteObject(iconinfo.hbmMask);
            if (iconinfo.hbmColor) DeleteObject(iconinfo.hbmColor);
        }

        DrawIcon(memoryDc_, pt.x, pt.y, hcur);
    }

    return bitmapToDIB(bitmapDc_, ppIBitMap);
}

// -----------------------------------------------------------------------------
HRESULT Grabber::bitmapToDIB(HBITMAP hBitmap, IBitMap** ppIBitMap) const
{
    HRESULT hr;
    // fill in BITMAP structure, return NULL if it didn't work 
    BITMAP bm;
    GetObject(hBitmap, sizeof(BITMAP), &bm);

    // calculate bits per pixel
    WORD biBits = bm.bmPlanes * bm.bmBitsPixel;

    // calcualte color depth
    if (colorDepth_)      { biBits = colorDepth_; }

    // force to valid values
    if (biBits <= 16)       { biBits = 16; }
    else if (biBits <= 24)  { biBits = 24; }
    else                    { biBits = 32; }

    // create bitmap object
    CComPtr<BitMap> pBitMap;
    hr = newBitMap(bm.bmWidth, bm.bmHeight, biBits, &pBitMap);
    if (FAILED(hr)) return hr;

    // get bitmap info header
    LPBITMAPINFOHEADER lpbi;
    hr = pBitMap->GetBitMapInfoHeader(&lpbi);
    if (FAILED(hr)) return hr;

    // and bitmap buffer
    LPBYTE lpBits;
    hr = pBitMap->GetBuffer(&lpBits);
    if (FAILED(hr)) return hr;

    // let the device driver convert the device-dependent bitmap (DDB) to a DIB
    if (!GetDIBits(desktopDc_, hBitmap, 0, (UINT)bm.bmHeight, lpBits, (LPBITMAPINFO)lpbi, DIB_RGB_COLORS))
        return CAMIT_HRESULT_FROM_WIN32(GetLastError());

    return pBitMap->GetUnknown()->QueryInterface(IID_IBitMap, (void**)ppIBitMap);
}

//------------------------------------------------------------------------------
HRESULT Grabber::newBitMap(LONG width, LONG height, WORD bits, BitMap** ppBitMap) const
{
    for (BitMapPool::iterator it = bitMaps_.begin(); it != bitMaps_.end(); ++it) 
    {

        if (SUCCEEDED((*it)->ReUse(width, height, bits))) 
        {

            // found reusable memory, discard others
            if (--spareBitmaps_ < 1) 
            {
                spareBitmaps_ = 1;
            }

            int grace = spareBitmaps_;
            for (BitMapPool::iterator it2 = bitMaps_.begin(); it2 != bitMaps_.end();) 
            {

                if (it2 != it) 
                {

                    HRESULT hr = (*it2)->ReUse(width, height, bits);

                    if ((hr == S_OK || hr == E_INVALIDARG) && (grace-- <= 0)) 
                    {
                        (*it2)->GetUnknown()->Release();
                        it2 = bitMaps_.erase(it2);
                        continue;
                    }
                }

                ++it2;
            }

            *ppBitMap = *it;
            (*ppBitMap)->AddRef();
            return S_OK;
        }
    }

    spareBitmaps_ += 4;

    *ppBitMap = new BitMap(width, height, bits);
    (*ppBitMap)->AddRef();

    bitMaps_.push_back(*ppBitMap);
    bitMaps_.back()->AddRef();

    return S_OK;
}

//------------------------------------------------------------------------------
size_t Grabber::getPoolSize() const
{
    return bitMaps_.size();
}

// -----------------------------------------------------------------------------
HCURSOR Grabber::fetchCursorHandle() const
{
    if (customCursor_)
        return customCursor_;
    else if (HCURSOR hCursor = getCurrentCursorFromHook())
        return hCursor;
    else
        return defaultCursor_;
}

// -----------------------------------------------------------------------------
void Grabber::hiliteCursor(HDC hDC, int xoffset, int yoffset) const
{
    int     fullSizeX = 128;
    int     fullSizeY = 128;

    double  x1 = 0.0, x2 = 0.0, y1 = 0.0, y2 = 0.0;

    // create a compatible device context and select a memory bitmap into it
    SmrtDcHandle      hDCBits(CreateCompatibleDC(hDC));
    SmrtBitMapHandle  bitmapDc_(CreateCompatibleBitmap(hDC, fullSizeX, fullSizeY));
    SmrtGdiSelect     hOldBitMap(hDCBits, SelectObject(hDCBits, bitmapDc_));

    switch (hiliteShape_)
    {
    case CH_CIRCLE:
    case CH_SQUARE:
        x1 = (fullSizeX - hiliteSize_) / 2.0;
        x2 = (fullSizeX + hiliteSize_) / 2.0;
        y1 = (fullSizeY - hiliteSize_) / 2.0;
        y2 = (fullSizeY + hiliteSize_) / 2.0;
        break;

    case CH_ELLIPSE:
    case CH_RECTANGLE:
        x1 = (fullSizeX - hiliteSize_) / 2.0;
        x2 = (fullSizeX + hiliteSize_) / 2.0;
        y1 = (fullSizeY - hiliteSize_ / 2.0) / 2.0;
        y2 = (fullSizeY + hiliteSize_ / 2.0) / 2.0;
        break;
    }

    HBRUSH          hBrushWhite = (HBRUSH)GetStockObject(WHITE_BRUSH);
    SmrtPenHandle   hNullPen(CreatePen(PS_NULL, 0, 0));
    SmrtBrushHandle hBrushHilite(CreateSolidBrush(hiliteColor_));

    SmrtGdiSelect   hOldBrush(hDCBits, SelectObject(hDCBits, hBrushWhite));
    SmrtGdiSelect   hOldPen(hDCBits, SelectObject(hDCBits, hNullPen));
    Rectangle(hDCBits, 0, 0, fullSizeX + 1, fullSizeY + 1);

    SmrtGdiSelect   hOldBrush2(hDCBits, SelectObject(hDCBits, hBrushHilite));
    switch (hiliteShape_)
    {
    case CH_CIRCLE:
    case CH_ELLIPSE:
        Ellipse(hDCBits, (int)x1, (int)y1, (int)x2, (int)y2);
        break;

    case CH_RECTANGLE:
    case CH_SQUARE:
        Rectangle(hDCBits, (int)x1, (int)y1, (int)x2, (int)y2);
        break;
    }

    // OffScreen Buffer
    BitBlt(hDC, xoffset, yoffset, fullSizeX, fullSizeY, hDCBits, 0, 0, SRCAND);
}

} // namespace camit
