//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "SelWnd.h"
#include "../res/resource.h"
#include "SmrtHandle.h"
#include "util.h"

// The current SDK has a bug in SetWindowLongPtr and issues invalid warnings
#pragma warning(disable: 4244 4312)

namespace camit {

//------------------------------------------------------------------------------
SelWnd::SelWnd(Mode mode, 
               int width /* = -1 */, 
               int height /* = -1 */, 
               int border /* = 2 */) :
    windowHandle_(NULL),
    regionMode_(mode),
    borderSize_(border),
    selectingRegion_(false)
{
    // initialize fixed region
    SetRect(&fixedArea_, 0, 0, width-1, height-1);
    SetRectEmpty(&recordedArea_);
    SetRectEmpty(&prevRecordedArea_);
    SetRectEmpty(&offset_);

    // get module instance handle
    moduleHandle_ = GetModuleHandle(NULL);

    // determine screen dimensions
    SmrtDcRelease hDc(NULL, GetDC(NULL));
    screenSizeX_ = GetDeviceCaps(hDc, HORZRES);
    screenSizeY_ = GetDeviceCaps(hDc, VERTRES);

    // register our windows class
    registerWndClass();
    create();
}

//------------------------------------------------------------------------------
ATOM SelWnd::registerWndClass()
{
    // register the window class
    WNDCLASSEX  wcex;
    ZeroMemory(&wcex, sizeof(wcex));
    wcex.cbSize = sizeof(wcex);
    wcex.style = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc = wndProcStub;
    wcex.cbClsExtra = 0;
    wcex.cbWndExtra = 0;
    wcex.hInstance = moduleHandle_;
    wcex.hIcon = NULL;
    wcex.hCursor = LoadCursor(NULL, IDC_CROSS);
    wcex.hbrBackground = NULL;
    wcex.lpszMenuName = NULL;
    wcex.lpszClassName = _T("CamItSelWnd");
    wcex.hIconSm = NULL;
    return RegisterClassEx(&wcex);
}

//------------------------------------------------------------------------------
void SelWnd::create()
{
    windowHandle_.reset(CreateWindowEx(WS_EX_TOPMOST | WS_EX_TOOLWINDOW, 
                _T("CamItSelWnd"), 
                _T(""),  
                WS_POPUP, 
                0, 0, 0, 0,
                NULL, 
                NULL, 
                moduleHandle_, 
                reinterpret_cast<LPVOID>(this)));

    if (regionMode_ == RM_FIXED) 
    {
        ShowWindow(windowHandle_, SW_MAXIMIZE);
        UpdateWindow(windowHandle_);
        initFixedWnd();
    }
    else if (regionMode_ == RM_VARIABLE)
    {
        ShowWindow(windowHandle_, SW_MAXIMIZE);
        UpdateWindow(windowHandle_);
    }
}

//------------------------------------------------------------------------------
bool SelWnd::run()
{
    if (windowHandle_.equals(NULL))
        return false;

    MSG msg;
    while (GetMessage(&msg, NULL, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    return true;
}

//------------------------------------------------------------------------------
LRESULT CALLBACK SelWnd::wndProcStub(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    SelWnd* pThis = NULL;

    if (message == WM_CREATE) 
    {
        CREATESTRUCT* cs = reinterpret_cast<CREATESTRUCT*>(lParam);
        pThis = reinterpret_cast<SelWnd*>(cs->lpCreateParams);
        SetWindowLongPtr(hWnd, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(pThis));
    }
    else
    {
        pThis = reinterpret_cast<SelWnd*>(GetWindowLongPtr(hWnd, GWLP_USERDATA));
    }

    if (pThis)
        return pThis->wndProc(hWnd, message, wParam, lParam);
    else
        return DefWindowProc(hWnd, message, wParam, lParam);
}

// -----------------------------------------------------------------------------
LRESULT SelWnd::wndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
    case WM_MOUSEMOVE:
        if (regionMode_ == RM_FIXED)
        {

            POINT pt;
            GetCursorPos(&pt);
            CopyRect(&recordedArea_, &offset_);
            OffsetRect(&recordedArea_, pt.x, pt.y);

            if (recordedArea_.left < 0) 
            {
                recordedArea_.left = 0;
                recordedArea_.right = ((fixedArea_.right) - (fixedArea_.left));
            }

            if (recordedArea_.top < 0) 
            {
                recordedArea_.top = 0;
                recordedArea_.bottom = ((fixedArea_.bottom) - (fixedArea_.top));
            }

            if (recordedArea_.right > screenSizeX_ )
            {
                recordedArea_.right = screenSizeX_ ;
                recordedArea_.left = screenSizeX_ - ((fixedArea_.right) - (fixedArea_.left));
            }

            if (recordedArea_.bottom > screenSizeY_) 
            {
                recordedArea_.bottom = screenSizeY_;
                recordedArea_.top = screenSizeY_ - ((fixedArea_.bottom) - (fixedArea_.top));
            }

            if (!EqualRect(&prevRecordedArea_, &recordedArea_)) 
            {
                SmrtDcRelease hDc(hWnd, GetDC(hWnd));
                drawSelectionFrame(hDc, FALSE, &prevRecordedArea_);  // erase old rubber-band
                drawSelectionFrame(hDc, TRUE, &recordedArea_);       // new rubber-band
            } // if old

            prevRecordedArea_ = recordedArea_;
        }
        else if (regionMode_ == RM_VARIABLE) 
        {
            if (selectingRegion_)
            {
                POINT pt;
                GetCursorPos(&pt);

                SmrtDcRelease hDc(hWnd, GetDC(hWnd));
                drawSelectionFrame(hDc, FALSE, &recordedArea_);  // erase old rubber-band

                SetRect(&recordedArea_, startPoint_.x, startPoint_.y, pt.x, pt.y);

                normalizeRect(recordedArea_);
                drawSelectionFrame(hDc, TRUE, &recordedArea_);   // new rubber-band
            }
        }

        return DefWindowProc(hWnd, message, wParam, lParam);

    case WM_LBUTTONUP: 
        {
            // erase final
            normalizeRect(recordedArea_);
            SmrtDcRelease hDc(hWnd, GetDC(hWnd));
            drawSelectionFrame(hDc, FALSE, &prevRecordedArea_);

            ShowWindow(hWnd, SW_HIDE);

            if (!IsRectEmpty(&recordedArea_)) 
            {
                normalizeRect(recordedArea_);
            }

            selectingRegion_ = false;

            PostQuitMessage(0);
            return DefWindowProc(hWnd, message, wParam, lParam); 
        }
        break;

    case WM_LBUTTONDOWN:
        // User pressed left button, initialize selection 
        // Set origin to current mouse position (in window coords)
        if (regionMode_ == RM_VARIABLE)
        {
            POINT pt;
            GetCursorPos(&pt);
            startPoint_ = pt;
            SetRect(&recordedArea_, pt.x, pt.y, pt.x, pt.y);

            normalizeRect(recordedArea_); // Make sure it is a normal rect
            SmrtDcRelease hDc(hWnd, GetDC(hWnd));
            drawSelectionFrame(hDc, TRUE, &recordedArea_);     // Draw the rubber-band box
            selectingRegion_ = true;
        }

        return DefWindowProc(hWnd, message, wParam, lParam);

    case WM_RBUTTONDOWN:

        if (regionMode_ == RM_FIXED) 
        {
            // erase final
            SmrtDcRelease hDc(hWnd, GetDC(hWnd));
            drawSelectionFrame(hDc, FALSE, &prevRecordedArea_);

            // Cancel the operation
            ShowWindow(hWnd, SW_HIDE);
        }

        return DefWindowProc(hWnd, message, wParam, lParam);

    case WM_DESTROY:
        PostQuitMessage(0);
        break;

    default:
        return DefWindowProc(hWnd, message, wParam, lParam);
    }

    return 0;
}

// -------------------------------------------------------------------------------
int SelWnd::initFixedWnd()
{
    SmrtDcRelease hDc(windowHandle_, GetDC(windowHandle_));

    RECT rcScreen = { 0, 0, screenSizeX_, screenSizeY_ };
    normalizeRect(fixedArea_);
    IntersectRect(&recordedArea_, &fixedArea_, &rcScreen);

    drawSelectionFrame(hDc, TRUE, &recordedArea_);
    prevRecordedArea_ = recordedArea_;

    // Set Curosr at the centre of the clip rectangle
    POINT ptOrigin;
    ptOrigin.x = (recordedArea_.right + recordedArea_.left) / 2;
    ptOrigin.y = (recordedArea_.top + recordedArea_.bottom) / 2;
    CopyRect(&offset_, &recordedArea_);
    OffsetRect(&offset_, -ptOrigin.x, -ptOrigin.y);

    return 0;
}

// -----------------------------------------------------------------------------
//  Draws the selected clip rectangle with its dimensions on the DC
// -----------------------------------------------------------------------------
void SelWnd::drawSelectionFrame(HDC hDc, BOOL fDraw, LPRECT lprClip)
{
    int   x, y, len, dx, dy;
    SIZE  sExtent;

    RECT rectDraw;
    CopyRect(&rectDraw, lprClip);
    if (!IsRectEmpty(&rectDraw))
    {
        // If a rectangular clip region has been selected, draw it
        SmrtBrushHandle newbrush(CreateHatchBrush(HS_BDIAGONAL, RGB(0, 0, 100)));
        SmrtGdiSelect   oldbrush(hDc, SelectObject(hDc, newbrush));

        // PatBlt SRCINVERT regardless fDraw is TRUE or FALSE
        PatBlt(hDc, rectDraw.left, rectDraw.top, rectDraw.right - rectDraw.left, borderSize_, PATINVERT);
        PatBlt(hDc, rectDraw.left, rectDraw.bottom - borderSize_, borderSize_, -(rectDraw.bottom - rectDraw.top - 2 * borderSize_), PATINVERT);
        PatBlt(hDc, rectDraw.right - borderSize_, rectDraw.top + borderSize_, borderSize_, rectDraw.bottom - rectDraw.top - 2 * borderSize_, PATINVERT);
        PatBlt(hDc, rectDraw.right, rectDraw.bottom - borderSize_, -(rectDraw.right - rectDraw.left), borderSize_, PATINVERT);

        SmrtDcHandle   hdcBits(CreateCompatibleDC(hDc));

        SmrtGdiHandle  newfont(GetStockObject(ANSI_VAR_FONT));
        SmrtGdiSelect  oldfont(hDc, SelectObject(hDc, newfont));

        TCHAR sz[80];
        wsprintf(sz,
            _T("Left : %d  Top : %d  Width : %d  Height : %d"),
            rectDraw.left,
            rectDraw.top,
            rectDraw.right - rectDraw.left,
            rectDraw.bottom - rectDraw.top);
        len = lstrlen(sz);
        DWORD dw = GetTextExtentPoint(hDc, sz, len, &sExtent);

        // dw = GetTextExtentPoint(hdcBits, sz, len, &sExtent);
        dx = sExtent.cx;
        dy = sExtent.cy;
        x = rectDraw.left + 10;

        if (rectDraw.top < (dy + borderSize_ + 2))
            y = rectDraw.bottom + borderSize_ + 2;
        else
            y = rectDraw.top - dy - borderSize_ - 2;

        if (fDraw) 
        {
            // Save Original Picture
            savedBitmap_.save(hDc, hdcBits, x - 4, y - 4, dx + 8, dy + 8);

            // Text
            COLORREF  oldtextcolor = SetTextColor(hDc, RGB(0, 0, 0));
            COLORREF  oldbkcolor = SetBkColor(hDc, RGB(255, 255, 255));
            SetBkMode(hDc, TRANSPARENT);

            RoundRect(hDc, x - 4, y - 4, x + dx + 4, y + dy + 4, 10, 10);

            SetBkMode(hDc, OPAQUE);

            ExtTextOut(hDc, x, y, 0, NULL, sz, len, NULL);
            SetBkColor(hDc, oldbkcolor);
            SetTextColor(hDc, oldtextcolor);
        }
        else 
        {
            savedBitmap_.restore(hDc, hdcBits);
        }

        // Icon
        if ((rectDraw.right - rectDraw.left - 10 > 35)
            && (rectDraw.bottom - rectDraw.top - 10 > dy + 40)) 
        {
            SmrtBitMapHandle hbv(LoadBitmap(moduleHandle_, MAKEINTRESOURCE(IDB_NOTEPAD)));
            SmrtGdiSelect    oldbitmap(hdcBits, SelectObject(hdcBits, hbv));
            BitBlt(hDc, rectDraw.left + 10, rectDraw.bottom - 42, 30, 32, hdcBits, 0, 0, SRCINVERT);
        }
    }
}

// -----------------------------------------------------------------------------
//  Highlight the window frame
// -----------------------------------------------------------------------------
RECT SelWnd::frameWindow()
{
    RECT  rectWin;
    RECT  rectFrame;

    SetRect(&rectWin, 0, 0, screenSizeX_, screenSizeY_);

    if (!IsWindow(windowHandle_)) return rectWin;

    SmrtDcRelease hDc(windowHandle_, GetWindowDC(windowHandle_));
    GetWindowRect(windowHandle_, &rectWin);

    rectFrame = rectWin;
    OffsetRect(&rectFrame, -rectFrame.left, -rectFrame.top);

    if (!IsRectEmpty(&rectFrame)) 
    {
        SmrtBrushHandle newbrush(CreateHatchBrush(HS_BDIAGONAL, RGB(0, 0, 100)));
        SmrtGdiSelect   oldbrush(hDc, SelectObject(hDc, newbrush));

        PatBlt(hDc, recordedArea_.left, recordedArea_.top, recordedArea_.right - recordedArea_.left, borderSize_, PATINVERT);
        PatBlt(hDc, recordedArea_.left, recordedArea_.bottom - borderSize_, borderSize_, -(recordedArea_.bottom - recordedArea_.top - 2 * borderSize_), PATINVERT);
        PatBlt(hDc, recordedArea_.right - borderSize_, recordedArea_.top + borderSize_, borderSize_, recordedArea_.bottom - recordedArea_.top - 2 * borderSize_, PATINVERT);
        PatBlt(hDc, recordedArea_.right, recordedArea_.bottom - borderSize_, -(recordedArea_.right - recordedArea_.left), borderSize_, PATINVERT);
    }

    return rectWin;
}


} // namespace camit
