//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include <stdafx.h>
#include "WMFile.h"
#include "SmrtHandle.h"
#include "WMProfiles.h"
#include "WMMediaTypePtr.h"
#include "ArgsInfo.h"
#include <IUnknownImpl.h>
#include <objbase.h>
#include <stdio.h>
#include <algorithm>

namespace camit {

#define VERBOSE_PRINT(x) if (verbose_) { _tprintf x; }

// WMV files contain several streams, identified by the "stream number".
// We use the following stream numbers (they can be anything from 1 through 63)
#define AUDIO_STREAM_NUMBER 1
#define VIDEO_STREAM_NUMBER 2

//------------------------------------------------------------------------------
// Status callback coclass, used by the IWMIndexer interface to index the
// recorded file. Indexing adds fast forward / rewind capabilities to the 
// file.
//------------------------------------------------------------------------------
class WMFile::StatusCallback : public IUnknownImpl,
                               public IWMStatusCallback
{
public:
    StatusCallback();

    // IUknown implementation
    DECLARE_IUNKNOWN;
    STDMETHODIMP  NonDelegatingQueryInterface(REFIID riid, void** ppv);

    // IWMStatusCallback
    STDMETHODIMP  OnStatus(WMT_STATUS        status,
                           HRESULT           hr,
                           WMT_ATTR_DATATYPE dwType,
                           BYTE*             pValue,
                           void*             pvContext);
};

// -----------------------------------------------------------------------------
WMFile::WMFile()
{
    verbose_ = ArgsInfo::getInstance().verbose_;
}

// -----------------------------------------------------------------------------
WMFile::~WMFile()
{
}

// -----------------------------------------------------------------------------
bool WMFile::testAlignment(Format& fmt) const
{
    HRESULT hr;

    // get profile manager
    CComPtr<IWMProfileManager> pIMgr;
    hr = WMCreateProfileManager(&pIMgr);
    if (FAILED(hr)) throw ComError(hr);

    // enable SDK 7.0 functionality
    CComPtr<IWMProfileManager2> pIMgr2;
    hr = pIMgr->QueryInterface(IID_IWMProfileManager2, (void**)&pIMgr2);
    if (FAILED(hr)) throw ComError(hr);

    pIMgr2->SetSystemProfileVersion(WMT_VER_7_0);

    // select codec based config
    WMProfiles prof(pIMgr);

    CComPtr<IWMStreamConfig> pICfg;
    prof.getVideoCodec(fmt.videoInfo_.bmiHeader.biCompression, &pICfg);

    CComPtr<IWMVideoMediaProps> pIMediaProps;
    hr = pICfg->QueryInterface(IID_IWMVideoMediaProps, (void**)&pIMediaProps);
    if (FAILED(hr)) throw ComError(hr);

    // configure media type
    WMMediaTypePtr mt(pIMediaProps);
    mt->bFixedSizeSamples = TRUE;
    mt->bTemporalCompression = FALSE;
    mt->lSampleSize = DIBSIZE(fmt.videoInfo_.bmiHeader);

    // copy video info header in media properties
    WMVIDEOINFOHEADER*  pvih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);
    CopyMemory(pvih, &fmt.videoInfo_, sizeof(WMVIDEOINFOHEADER));

    // update media properties with media type
    return SUCCEEDED(pIMediaProps->SetMediaType(mt));
}

// -----------------------------------------------------------------------------
bool WMFile::open(TCHAR* strFileName, Format& fmt, bool bDummySound)
{
    HRESULT hr;

    // save arguments
    fileName_ = strFileName;
    dummySound_ = bDummySound;

    // get profile manager
    CComPtr<IWMProfileManager> pIMgr;
    hr = WMCreateProfileManager(&pIMgr);
    if (FAILED(hr)) throw ComError(hr);

    // enable SDK 7.0 functionality
    CComPtr<IWMProfileManager2> pIMgr2;
    hr = pIMgr->QueryInterface(IID_IWMProfileManager2, (void**)&pIMgr2);
    if (FAILED(hr)) throw ComError(hr);

    pIMgr2->SetSystemProfileVersion(WMT_VER_7_0);

    // create empty profile
    CComPtr<IWMProfile> pIProfile;
    hr = pIMgr->CreateEmptyProfile(WMT_VER_7_0, &pIProfile);
    if (FAILED(hr)) throw ComError(hr);

    // create audio stream and add to profile
    CComPtr<IWMStreamConfig> pICfgAudio;
    hr = createAudioStreamConfig(pIMgr, AUDIO_STREAM_NUMBER, fmt, &pICfgAudio);
    if (FAILED(hr)) throw ComError(hr);
    if (bDummySound) 
    {
        pICfgAudio->SetBitrate(0);  // WMV requires audio, but we can set it to 0 bits/sec
    }

    pIProfile->AddStream(pICfgAudio);

    // create video stream and add to profile
    CComPtr<IWMStreamConfig> pICfgVideo;
    hr = createVideoStreamConfig(pIMgr, VIDEO_STREAM_NUMBER, fmt, &pICfgVideo);
    if (FAILED(hr)) throw ComError(hr);

    pIProfile->AddStream(pICfgVideo);

    if (verbose_) 
    {
        WMProfiles(pIMgr).dumpProfile(pIProfile);
    }

    // create writer object
    hr = WMCreateWriter(NULL, &pIWriter_);
    if (FAILED(hr)) throw ComError(hr);

    // set profile
    pIWriter_->SetProfile(pIProfile);

    // specify filename
    pIWriter_->SetOutputFilename(A2W(fileName_.c_str()));

    // advanced writer configuration
    hr = pIWriter_->QueryInterface(IID_IWMWriterAdvanced, (void**)&pIWriterAdvanced_);
    if (FAILED(hr)) throw ComError(hr);

    pIWriterAdvanced_->SetLiveSource(TRUE);

    // get the video & audio stream 
    DWORD inputCnt;
    hr = pIWriter_->GetInputCount(&inputCnt);
    if (FAILED(hr)) throw ComError(hr);

    for (DWORD iInput = 0; iInput < inputCnt; ++iInput) 
    {
        CComPtr<IWMInputMediaProps> pIProps;
        hr = pIWriter_->GetInputProps(iInput, &pIProps);
        if (FAILED(hr)) throw ComError(hr);

        GUID guidType;
        hr = pIProps->GetType(&guidType);
        if (FAILED(hr)) throw ComError(hr);

        if (guidType == WMMEDIATYPE_Video) 
        {
            videoInput_ = iInput;
            configureVideoInput(fmt, pIWriter_, videoInput_, pIProps);
        }
        else if (guidType == WMMEDIATYPE_Audio) 
        {
            audioInput_ = iInput;
            configureAudioInput(fmt, pIWriter_, audioInput_, pIProps);
        }
    }

    // set clip attributes
    setHeaderInfo(L"Title", A2W(fmt.clipTitle_.c_str()));
    setHeaderInfo(L"Author", A2W(fmt.clipAuthor_.c_str()));
    setHeaderInfo(L"Copyright", A2W(fmt.clipCopyright_.c_str()));
    setHeaderInfo(L"Description", A2W(fmt.clipDescription_.c_str()));

    // reset previous key-frame counter
    keyFrameIntervalMs_ = 1000 * fmt.videoSecPerKey_;
    key();

    discontinuity_ = false;

    // start writing
    pIWriter_->BeginWriting();

    // allocate dummy sound buffer
    if (dummySound_) 
    {
        hr = pIWriter_->AllocateSample(8000 / fmt.getVideoFPS(), &dummySoundBuffer_);
        if (FAILED(hr)) throw ComError(hr);

        // zero memory to silence
        LPBYTE buffer;
        DWORD len;
        hr = dummySoundBuffer_->GetBufferAndLength(&buffer, &len);
        if (FAILED(hr)) throw ComError(hr);
        ZeroMemory(buffer, len);
    }

    VERBOSE_PRINT((_T("Begin Writing\n")));
    return true;
}

// -----------------------------------------------------------------------------
bool WMFile::writeVideoSample(INSSBuffer* pIBmp, DWORD timeMs)
{
    // flags
    DWORD flags = 0;

    // check if key frame
    bool key = ((static_cast<LONG>(timeMs) - prevKeyFrameMs_) > (LONG)keyFrameIntervalMs_);

    if (key) 
    {
        flags |= WM_SF_CLEANPOINT;
    }

    if (discontinuity_) 
    {
        flags |= WM_SF_DISCONTINUITY;
    }

    // calculate frame time
    QWORD timeFrame = static_cast<QWORD>(timeMs) * UINT64_C(10000);

    if (verbose_) 
    {
        VERBOSE_PRINT((_T("Frame at %d ms\n"), timeMs));
    }

    // write frame
    HRESULT hr = pIWriter_->WriteSample(videoInput_, timeFrame, flags, pIBmp);

    if (dummySound_) 
    {
        writeAudioSample(dummySoundBuffer_, timeMs);
    }

    if (   hr == NS_E_LATE_OPERATION
        || hr == NS_E_TOO_MUCH_DATA) 
    {
        VERBOSE_PRINT((_T("Lost frame!\n")));
        return false;
    }
    else if (FAILED(hr))
        throw ComError(hr);

    if (key) 
    {
        prevKeyFrameMs_ = timeMs;
    }

    discontinuity_ = false;

    return true;
}

// -----------------------------------------------------------------------------
bool WMFile::writeAudioSample(INSSBuffer* pISample, DWORD timeMs)
{
    // calculate frame time
    QWORD timeSample = static_cast<QWORD>(timeMs) * UINT64_C(10000);

    if (verbose_) 
    {
        VERBOSE_PRINT((_T("Audio at %d ms\n"), timeMs));
    }

    // write sample
    HRESULT hr = pIWriter_->WriteSample(audioInput_, timeSample, 0, pISample);

    if (   hr == NS_E_LATE_OPERATION
        || hr == NS_E_TOO_MUCH_DATA) 
    {
        VERBOSE_PRINT((_T("Lost audio sample!\n")));
        return false;
    }
    else if (FAILED(hr))
        throw ComError(hr);

    return true;
}

// -----------------------------------------------------------------------------
void WMFile::key()
{
    prevKeyFrameMs_ = -1000000;
}

// -----------------------------------------------------------------------------
void WMFile::discontinuity()
{
    discontinuity_ = true;
}

// -----------------------------------------------------------------------------
void WMFile::close()
{
    HRESULT hr;

    pIWriter_->EndWriting();

    if (verbose_) 
    {

        WM_WRITER_STATISTICS stats;
        pIWriterAdvanced_->GetStatistics(AUDIO_STREAM_NUMBER, &stats);
        _tprintf(_T("Audio: cnt=%I64d bytes=%I64d dropped=%I64d bps=%d bps_a=%d bps_e=%d\n"),
            stats.qwSampleCount, 
            stats.qwByteCount, 
            stats.qwDroppedByteCount,
            stats.dwCurrentBitrate, 
            stats.dwAverageBitrate,
            stats.dwExpectedBitrate);

        pIWriterAdvanced_->GetStatistics(VIDEO_STREAM_NUMBER, &stats);
        _tprintf(_T("Video: cnt=%I64d bytes=%I64d dropped=%I64d bps=%d bps_a=%d bps_e=%d\n"),
            stats.qwSampleCount, 
            stats.qwByteCount, 
            stats.qwDroppedByteCount,
            stats.dwCurrentBitrate, 
            stats.dwAverageBitrate,
            stats.dwExpectedBitrate);


        VERBOSE_PRINT((_T("Finished writing\n")));

    }

    pIWriter_.Detach()->Release();

    // index file
    CComPtr<IWMIndexer> pIIndexer;
    hr = WMCreateIndexer(&pIIndexer);
    if (FAILED(hr)) throw ComError(hr);

    SmrtWin32Handle hEvent(CreateEvent(NULL, TRUE, FALSE, NULL));
    CComPtr<IWMStatusCallback> pIStatusCallback = new StatusCallback();
    pIIndexer->StartIndexing(A2W(fileName_.c_str()), pIStatusCallback, reinterpret_cast<void*>(hEvent.get()));

    // wait until finished
    WaitForSingleObject(hEvent, INFINITE);
}

// -----------------------------------------------------------------------------
HRESULT WMFile::createVideoStreamConfig(IWMProfileManager*  pIMgr,
                                        WORD                dwStreamNumber,
                                        Format&             fmt,
                                        IWMStreamConfig**   ppIWMStreamConfig) const
{
    HRESULT hr;

    if (!ppIWMStreamConfig)
        return E_POINTER;

    *ppIWMStreamConfig = NULL;

    // select codec based config
    WMProfiles prof(pIMgr);
    CComPtr<IWMStreamConfig> pICfg;
    prof.getVideoCodec(fmt.videoInfo_.bmiHeader.biCompression, &pICfg);

    CComPtr<IWMVideoMediaProps> pIMediaProps;
    hr = pICfg->QueryInterface(IID_IWMVideoMediaProps, (void**)&pIMediaProps);
    if (FAILED(hr)) return hr;

    // configure media type
    WMMediaTypePtr mt(pIMediaProps);

    // copy video info header in media properties
    WMVIDEOINFOHEADER* pvih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);
    CopyMemory(pvih, &fmt.videoInfo_, sizeof(WMVIDEOINFOHEADER));

    // update media properties with media type
    pIMediaProps->SetMediaType(mt);

    // video media properties
    pIMediaProps->SetQuality(fmt.videoQuality_);
    pIMediaProps->SetMaxKeyFrameSpacing(INT64_C(10000000) * ((LONGLONG) fmt.videoSecPerKey_));

    // stream properties
    pICfg->SetStreamNumber(dwStreamNumber);
    pICfg->SetStreamName(L"Video Stream");
    pICfg->SetConnectionName(L"Video");
    pICfg->SetBitrate(fmt.videoInfo_.dwBitRate);

    *ppIWMStreamConfig = pICfg.Detach();
    return S_OK;
}

// -----------------------------------------------------------------------------
void WMFile::configureVideoInput(Format&               fmt,
                                 IWMWriter*            pIWriter,
                                 DWORD                 dwVideoInput,
                                 IWMInputMediaProps*   pIInputMediaProps) const
{
    // configure media type
    WMMediaTypePtr mt(pIInputMediaProps);

    // first set the VIDEOINFOHEADER
    WMVIDEOINFOHEADER* pvih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);
    CopyMemory(pvih, &fmt.videoInfo_, sizeof(WMVIDEOINFOHEADER));
    pvih->bmiHeader.biCompression = BI_RGB; // input is uncompressed

    mt->majortype             = WMMEDIATYPE_Video;
    mt->subtype               = getBitmapSubtype(&pvih->bmiHeader);
    mt->bFixedSizeSamples     = TRUE;
    mt->bTemporalCompression  = FALSE;
    mt->lSampleSize           = DIBSIZE(fmt.videoInfo_.bmiHeader);
    mt->formattype            = WMFORMAT_VideoInfo;
    mt->pUnk                  = NULL;


    // update media properties with media type
    pIInputMediaProps->SetMediaType(mt);

    // ... and apply to stream
    pIWriter->SetInputProps(dwVideoInput, pIInputMediaProps);
}

// -----------------------------------------------------------------------------
HRESULT WMFile::createAudioStreamConfig(IWMProfileManager*  pIMgr,
                                        WORD                dwStreamNumber,
                                        Format&             fmt,
                                        IWMStreamConfig**   ppIWMStreamConfig) const
{
    if (!ppIWMStreamConfig)
        return E_POINTER;
    *ppIWMStreamConfig = NULL;

    // select codec based configs
    WMProfiles prof(pIMgr);
    CComPtr<IWMStreamConfig> pICfg;
    prof.getAudioCodec(fmt.waveFormat_, fmt.audioCodecGuid_, &pICfg);

    // stream properties
    pICfg->SetStreamNumber(dwStreamNumber);
    pICfg->SetStreamName(L"Audio Stream");
    pICfg->SetConnectionName(L"Audio");

    *ppIWMStreamConfig = pICfg.Detach();
    return S_OK;
}

// -----------------------------------------------------------------------------
void WMFile::configureAudioInput(Format&               fmt,
                                 IWMWriter*            pIWriter,
                                 DWORD                 dwAudioInput,
                                 IWMInputMediaProps*   pIInputMediaProps) const
{
    LPWAVEFORMATEX  pwfe = &fmt.waveFormatRecorder_;

    if (dummySound_) 
    {
        WAVEFORMATEX  wfe;
        wfe.wFormatTag        = WAVE_FORMAT_PCM;
        wfe.nChannels         = 1;
        wfe.nSamplesPerSec    = 8000;
        wfe.wBitsPerSample    = 8;
        wfe.wFormatTag        = WAVE_FORMAT_PCM;
        wfe.nBlockAlign       = wfe.nChannels * wfe.wBitsPerSample / 8;
        wfe.nAvgBytesPerSec   = wfe.nSamplesPerSec * wfe.nBlockAlign;
        wfe.cbSize            = 0;
        pwfe = &wfe;
    }

    // configure media type
    WMMediaTypePtr mt(pIInputMediaProps);
    mt->majortype             = WMMEDIATYPE_Audio;
    mt->subtype               = WMMEDIASUBTYPE_PCM;
    mt->bFixedSizeSamples     = FALSE;
    mt->bTemporalCompression  = FALSE;
    mt->lSampleSize           = pwfe->nAvgBytesPerSec / fmt.getAudioBuffersPerSec();
    mt->formattype            = WMFORMAT_WaveFormatEx;
    mt->pUnk                  = NULL;
    mt->cbFormat              = sizeof(WAVEFORMATEX) + pwfe->cbSize;

    CopyMemory(mt->pbFormat, pwfe, sizeof(WAVEFORMATEX));

    // update media properties with media type
    pIInputMediaProps->SetMediaType(mt);

    // ... and apply to stream
    pIWriter->SetInputProps(dwAudioInput, pIInputMediaProps);
}

// -----------------------------------------------------------------------------
void WMFile::setHeaderInfo(WCHAR* name, const WCHAR* value)
{
    HRESULT hr;

    if (!name || !value) 
        return;

    CComPtr<IWMHeaderInfo> pIHeader;
    hr = pIWriter_->QueryInterface(IID_IWMHeaderInfo, (void**)&pIHeader);
    if (FAILED(hr)) throw ComError(hr);

    WORD bytes = static_cast<WORD>((wcslen(value) + 1) * sizeof(WCHAR));
    pIHeader->SetAttribute(0, name, WMT_TYPE_STRING, reinterpret_cast<BYTE*>(const_cast<WCHAR*>(value)), bytes);
}

// -----------------------------------------------------------------------------
WMFile::StatusCallback::StatusCallback()
{
}

// -----------------------------------------------------------------------------
STDMETHODIMP WMFile::StatusCallback::NonDelegatingQueryInterface(REFIID  riid, void**  ppv)
{
    if (riid == IID_IWMStatusCallback) 
    {
        *ppv = static_cast<IWMStatusCallback*>(this);
        static_cast<IWMStatusCallback*>(this)->AddRef();
        return S_OK;
    }
    else 
    {
        return IUnknownImpl::NonDelegatingQueryInterface(riid, ppv);
    }
}

// -----------------------------------------------------------------------------
STDMETHODIMP WMFile::StatusCallback::OnStatus(WMT_STATUS        status,
                                              HRESULT           hr,
                                              WMT_ATTR_DATATYPE dwType,
                                              BYTE*             pValue,
                                              void*             pvContext)
{
    switch (status)
    {
    case WMT_STARTED:           break;
    case WMT_INDEX_PROGRESS:    break;
    case WMT_ERROR:             break;
    case WMT_CLOSED:            SetEvent(reinterpret_cast<HANDLE>(pvContext)); break;
    }

    return S_OK;
}

//------------------------------------------------------------------------------
GUID WMFile::getBitmapSubtype(const BITMAPINFOHEADER* pbmiHeader) const
{
    static const GUID MEDIASUBTYPE_RGB1 =   { 0xe436eb78, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB4 =   { 0xe436eb79, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB8 =   { 0xe436eb7a, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB565 = { 0xe436eb7b, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB555 = { 0xe436eb7c, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB24 =  { 0xe436eb7d, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };
    static const GUID MEDIASUBTYPE_RGB32 =  { 0xe436eb7e, 0x524f, 0x11ce, { 0x9f, 0x53, 0x00, 0x20, 0xaf, 0x0b, 0xa7, 0x70 } };

    // If it's not RGB then create a GUID from the compression type
    if (pbmiHeader->biCompression != BI_RGB) 
    {
        if (pbmiHeader->biCompression != BI_BITFIELDS)
        {
            // DirectShow SDK: 
            // "Multimedia format types are marked with DWORDs built from four 8-bit
            //  chars and known as FOURCCs. New multimedia AM_MEDIA_TYPE definitions include
            //  a subtype GUID. In order to simplify the mapping, GUIDs in the range:
            //    XXXXXXXX-0000-0010-8000-00AA00389B71
            //  are reserved for FOURCCs."
            GUID guid = { pbmiHeader->biCompression, 0x0000, 0x0010, { 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71 } };
            return guid;
        }
    }

    // Map the RGB DIB bit depth to a image GUID
    switch(pbmiHeader->biBitCount) 
    {
        case 1    :   return MEDIASUBTYPE_RGB1;
        case 4    :   return MEDIASUBTYPE_RGB4;
        case 8    :   return MEDIASUBTYPE_RGB8;
        case 24   :   return MEDIASUBTYPE_RGB24;
        case 32   :   return MEDIASUBTYPE_RGB32;
        case 16   :
            {
                if (pbmiHeader->biCompression == BI_RGB)
                    return MEDIASUBTYPE_RGB555;

                BITMAPINFO* pbmInfo = (BITMAPINFO*)pbmiHeader;
                DWORD* pMask = (DWORD*)pbmInfo->bmiColors;

                static const DWORD bits555[] = {0x007C00, 0x0003E0, 0x00001F};
                if (std::mismatch(pMask, pMask + 3, bits555).first == pMask + 3)
                    return MEDIASUBTYPE_RGB555;

                static const DWORD bits565[] = {0x00F800, 0x0007E0, 0x00001F};
                if (std::mismatch(pMask, pMask + 3, bits565).first == pMask + 3)
                    return MEDIASUBTYPE_RGB565;
            }
            break;
    }

    return GUID_NULL;
}

} // namespace camit
