//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "WMProfiles.h"
#include "WMMediaTypePtr.h"
// #include <uuids.h>
#include <stdio.h>
#include <vector>

namespace camit {

//------------------------------------------------------------------------------
WMProfiles::WMProfiles()
{
    HRESULT hr;

    // get profile manager
    hr = WMCreateProfileManager(&profileMgr_);
    if (FAILED(hr)) throw ComError(hr);

    // enable SDK 7.0 functionality
    CComPtr<IWMProfileManager2> pIMgr2;
    hr = profileMgr_->QueryInterface(IID_IWMProfileManager2, (void**)&pIMgr2);
    if (FAILED(hr)) throw ComError(hr);

    pIMgr2->SetSystemProfileVersion(WMT_VER_7_0);
}

//------------------------------------------------------------------------------
WMProfiles::WMProfiles(IWMProfileManager* pIMgr) :
    profileMgr_(pIMgr)
{
}

//------------------------------------------------------------------------------
WMProfiles::~WMProfiles()
{
}

//------------------------------------------------------------------------------
void WMProfiles::getVideoCodec(FOURCC fourCc, IWMStreamConfig** ppIWMStreamConfig) const
{
    HRESULT hr;

    // get number of codecs
    CComPtr<IWMCodecInfo> pICodecInfo;
    hr = profileMgr_->QueryInterface(IID_IWMCodecInfo, (void**)&pICodecInfo);
    if (FAILED(hr)) throw ComError(hr);

    DWORD codecCnt;
    hr = pICodecInfo->GetCodecInfoCount(WMMEDIATYPE_Video, &codecCnt);
    if (FAILED(hr)) throw ComError(hr);

    // enumerate codecs
    for (unsigned iCodec = 0; iCodec < codecCnt; ++iCodec)
    {
        // get number of formats supported by codec
        DWORD fmtCnt;
        hr = pICodecInfo->GetCodecFormatCount(WMMEDIATYPE_Video, iCodec, &fmtCnt);
        if (FAILED(hr)) throw ComError(hr);

        // enumerate over formats
        for (DWORD iFormat = 0; iFormat < fmtCnt; ++iFormat) 
        {
            // get format
            CComPtr<IWMStreamConfig> pICfg;
            hr = pICodecInfo->GetCodecFormat(WMMEDIATYPE_Video, iCodec, iFormat, &pICfg);
            if (FAILED(hr)) throw ComError(hr);

            // get format properties
            CComPtr<IWMVideoMediaProps> pIMediaProps;
            hr = pICfg->QueryInterface(IID_IWMVideoMediaProps, (void**)&pIMediaProps);
            if (FAILED(hr)) throw ComError(hr);

            // get media type structure
            WMMediaTypePtr mt(pIMediaProps);

            if (mt->formattype != WMFORMAT_VideoInfo)
                continue;

            // extract video info header
            WMVIDEOINFOHEADER* vih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);

            // compare FOURCC
            if (vih->bmiHeader.biCompression == fourCc)
            {
                *ppIWMStreamConfig = pICfg.Detach();
                return;
            }
        }
    }

    if (FAILED(hr)) throw ComError(NS_E_VIDEO_CODEC_NOT_INSTALLED);
}

//------------------------------------------------------------------------------
void WMProfiles::getAudioCodec(const WAVEFORMATEX& wfe, REFGUID guidCodec, IWMStreamConfig** ppIWMStreamConfig) const
{
    static const GUID FORMAT_WaveFormatEx = 
        { 0x05589f81, 0xc356, 0x11ce, { 0xbf, 0x01, 0x00, 0xaa, 0x00, 0x55, 0x59, 0x5a } };

    HRESULT hr;

    // get number of codecs
    CComPtr<IWMCodecInfo> pICodecInfo;
    hr = profileMgr_->QueryInterface(IID_IWMCodecInfo, (void**)&pICodecInfo);
    if (FAILED(hr)) throw ComError(hr);

    DWORD codecCnt;
    hr = pICodecInfo->GetCodecInfoCount(WMMEDIATYPE_Audio, &codecCnt);
    if (FAILED(hr)) throw ComError(hr);

    // make local copy of wfe
    WAVEFORMATEX wfeLocal;
    CopyMemory(&wfeLocal, &wfe, sizeof(WAVEFORMATEX));

    // closest match
    unsigned iBestCodec = (unsigned)-1;
    unsigned iBestFormat = (unsigned)-1;
    DWORD    dwBestSamplingDelta = 100000;
    bool     bPickAny = false;

    while (true) 
    {
        // enumerate codecs
        for (unsigned iCodec = 0; iCodec < codecCnt; ++iCodec) 
        {
            // get number of formats supported by codec
            DWORD fmtCnt;
            hr = pICodecInfo->GetCodecFormatCount(WMMEDIATYPE_Audio, iCodec, &fmtCnt);
            if (FAILED(hr)) throw ComError(hr);

            // enumerate over formats
            for (DWORD iFormat = 0; iFormat < fmtCnt; ++iFormat) 
            {
                // get format
                CComPtr<IWMStreamConfig> pICfg;
                hr = pICodecInfo->GetCodecFormat(WMMEDIATYPE_Audio, iCodec, iFormat, &pICfg);
                if (FAILED(hr)) throw ComError(hr);

                // get format properties
                CComPtr<IWMMediaProps> pIMediaProps;
                hr = pICfg->QueryInterface(IID_IWMMediaProps, (void**)&pIMediaProps);
                if (FAILED(hr)) throw ComError(hr);

                // get media type structure
                WMMediaTypePtr mt(pIMediaProps);
                if (mt->formattype != FORMAT_WaveFormatEx)
                    continue;

                if (mt->subtype != guidCodec)
                    continue;

                // extract video info header
                WAVEFORMATEX* pwfe = reinterpret_cast<WAVEFORMATEX*>(mt->pbFormat);

                // compare with requested format
                if (bPickAny 
                    || (   pwfe->nSamplesPerSec >= wfeLocal.nSamplesPerSec
                        && (pwfe->wBitsPerSample == 0 || pwfe->wBitsPerSample == wfeLocal.wBitsPerSample)
                        && pwfe->nChannels == wfeLocal.nChannels))
                {
                    DWORD dwSamplingDelta = pwfe->nSamplesPerSec - wfeLocal.nSamplesPerSec;
                    if (bPickAny || dwSamplingDelta < dwBestSamplingDelta) 
                    {
                        iBestCodec = iCodec;
                        iBestFormat = iFormat;
                        dwBestSamplingDelta = dwSamplingDelta;
                    }
                }
            }
        }

        if (iBestCodec != -1) 
        {
            hr = pICodecInfo->GetCodecFormat(WMMEDIATYPE_Audio, iBestCodec, iBestFormat, ppIWMStreamConfig);
            if (FAILED(hr)) throw ComError(hr);
            return;
        }

        // no suitable compressor, try 8 bits
        if (wfe.wBitsPerSample == 16 && wfeLocal.wBitsPerSample == 16) 
        {
            wfeLocal.wBitsPerSample = 8;
            continue;
        }

        // no suitable compressor, try 16 bits
        if (wfe.wBitsPerSample == 8 && wfeLocal.wBitsPerSample == 8) 
        {
            wfeLocal.wBitsPerSample = 16;
            continue;
        }

        // no suitable compressor, try mono
        if (wfe.nChannels == 2 && wfeLocal.nChannels == 2) 
        {
            wfeLocal.nChannels = 1;
            continue;
        }

        // no suitable compressor, try stereo
        if (wfe.nChannels == 1 && wfeLocal.nChannels == 1) 
        {
            wfeLocal.nChannels = 2;
            continue;
        }

        // lost hope
        bPickAny = true;
    }

    throw ComError(NS_E_VIDEO_CODEC_NOT_INSTALLED);
}

//------------------------------------------------------------------------------
void WMProfiles::dumpSystemProfiles() const
{
    HRESULT hr;

    // enumerate profiles
    _tprintf( _T( "  Profile Indexes are as follows:\n" ) );

    DWORD profileCnt;
    hr = profileMgr_->GetSystemProfileCount(&profileCnt);
    if (FAILED(hr)) throw ComError(hr);

    for (unsigned iProfiles = 0; iProfiles < profileCnt; ++iProfiles) 
    {
        // load the profile
        CComPtr<IWMProfile> pIProfile;
        hr = profileMgr_->LoadSystemProfile(iProfiles, &pIProfile);
        if (FAILED(hr)) throw ComError(hr);

        // get name
        DWORD len = 0;
        hr = pIProfile->GetName(NULL, &len);
        if (FAILED(hr)) throw ComError(hr);

        std::vector<WCHAR> buf(len);
        hr = pIProfile->GetName(&buf[0], &len);
        if (FAILED(hr)) throw ComError(hr);

        _tprintf( _T("   %2d - %ws \n" ), iProfiles + 1, (WCHAR*)&buf[0]);
        dumpProfile(pIProfile);
    }
}

//------------------------------------------------------------------------------
void WMProfiles::dumpProfile(IWMProfile* pIProfile) const
{
    HRESULT hr;

    // number of streams
    DWORD streamCnt;
    hr = pIProfile->GetStreamCount(&streamCnt);
    if (FAILED(hr)) throw ComError(hr);

    for (unsigned iStream = 0; iStream < streamCnt; ++iStream) 
    {
        CComPtr<IWMStreamConfig> pICfg;
        hr = pIProfile->GetStream(iStream, &pICfg);
        if (FAILED(hr)) throw ComError(hr);

        // get name
        WORD len = 0;
        hr = pICfg->GetStreamName(NULL, &len);
        if (FAILED(hr)) throw ComError(hr);

        std::vector<WCHAR> buf(len);
        hr = pICfg->GetStreamName(&buf[0], &len);
        if (FAILED(hr)) throw ComError(hr);

        _tprintf(_T("         stream %d - %ws / "), iStream, &buf[0]);
        dumpCodecInfo(pICfg);
        _tprintf(_T("\n"));
    }
}

//------------------------------------------------------------------------------
void WMProfiles::dumpCodecs(REFGUID guidType) const
{
    HRESULT hr;

    // enumerate codecs
    CComPtr<IWMCodecInfo> pICodecInfo;
    hr = profileMgr_->QueryInterface(IID_IWMCodecInfo, (void**)&pICodecInfo);
    if (FAILED(hr)) throw ComError(hr);

    CComPtr<IWMCodecInfo2> pICodecInfo2;
    hr = profileMgr_->QueryInterface(IID_IWMCodecInfo2, (void**)&pICodecInfo2);
    if (FAILED(hr)) throw ComError(hr);

    DWORD codecCnt;
    hr = pICodecInfo->GetCodecInfoCount(guidType, &codecCnt);
    if (FAILED(hr)) throw ComError(hr);

    for (DWORD iCodec = 0; iCodec < codecCnt; ++iCodec) 
    {

        DWORD fmtCnt;
        hr = pICodecInfo->GetCodecFormatCount(guidType, iCodec, &fmtCnt);
        if (FAILED(hr)) throw ComError(hr);

        if (fmtCnt == 0) 
            continue;

        // get codec name
        DWORD len = 0;
        hr = pICodecInfo2->GetCodecName(guidType, iCodec, NULL, &len);
        if (FAILED(hr)) throw ComError(hr);

        std::vector<WCHAR> buf(len);
        hr = pICodecInfo2->GetCodecName(guidType, iCodec, &buf[0], &len);
        if (FAILED(hr)) throw ComError(hr);

        if (guidType == WMMEDIATYPE_Video) 
        {
            CComPtr<IWMStreamConfig> pICfg;
            hr = pICodecInfo->GetCodecFormat(guidType, iCodec, 0, &pICfg);
            if (FAILED(hr)) throw ComError(hr);

            dumpCodecInfo(pICfg);
            _tprintf(_T(" : %ws\n"), &buf[0]);
        }
        else if (guidType == WMMEDIATYPE_Audio) 
        {
            _tprintf(_T("%2d : %ws\n"), iCodec, &buf[0]);
        }
    }
}

//------------------------------------------------------------------------------
void WMProfiles::dumpCodecInfo(IWMStreamConfig* pICfg) const
{
    HRESULT hr;

    CComPtr<IWMMediaProps> pIMediaProps;
    hr = pICfg->QueryInterface(IID_IWMMediaProps, (void**)&pIMediaProps);
    if (FAILED(hr)) throw ComError(hr);

    // get media type
    WMMediaTypePtr mt(pIMediaProps);

    if (mt->formattype == WMFORMAT_VideoInfo) 
    {
        // video formats
        WMVIDEOINFOHEADER* vih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);
        DWORD fourCC = vih->bmiHeader.biCompression;
        _tprintf(_T("%c%c%c%c"), (BYTE)fourCC,
            (BYTE)(fourCC >> 8),
            (BYTE)(fourCC >> 16),
            (BYTE)(fourCC >> 24));
    }
    else if (mt->formattype == WMFORMAT_WaveFormatEx) 
    {
        // audio formats
        WAVEFORMATEX* wfe = reinterpret_cast<WAVEFORMATEX*>(mt->pbFormat);
        _tprintf(_T("%6.3f kHz, %s %2d-bit"),  (double)wfe->nSamplesPerSec/1000.0,
            wfe->nChannels == 1 ? _T("mono,  ") : _T("stereo,"),
            wfe->wBitsPerSample);
    }
}

//------------------------------------------------------------------------------
void WMProfiles::dumpMediaType(WM_MEDIA_TYPE* mt) const
{
    _tprintf(_T("WM_MEDIA_TYPE:\n"));

    _tprintf(_T("  majortype:     "));
    if      (mt->majortype == WMMEDIATYPE_Video)   _tprintf(_T("WMMEDIATYPE_Video (vids)\n"));
    else if (mt->majortype == WMMEDIATYPE_Audio)   _tprintf(_T("WMMEDIATYPE_Audio (auds)\n"));
    else if (mt->majortype == WMMEDIATYPE_Script)  _tprintf(_T("WMMEDIATYPE_Script (scmd)\n"));
    else if (mt->majortype == WMMEDIATYPE_Image)   _tprintf(_T("WMMEDIATYPE_Image\n"));
    else                                            _tprintf(_T("Unknown\n"));

    _tprintf(_T("  subtype:       "));
    if      (mt->subtype == WMMEDIASUBTYPE_Base)  _tprintf(_T("WMMEDIASUBTYPE_Base\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB1)  _tprintf(_T("WMMEDIASUBTYPE_RGB1\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB4)  _tprintf(_T("WMMEDIASUBTYPE_RGB4\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB8)  _tprintf(_T("WMMEDIASUBTYPE_RGB8\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB565)_tprintf(_T("WMMEDIASUBTYPE_RGB565\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB555)_tprintf(_T("WMMEDIASUBTYPE_RGB555\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB24) _tprintf(_T("WMMEDIASUBTYPE_RGB24\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_RGB32) _tprintf(_T("WMMEDIASUBTYPE_RGB32\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_I420)  _tprintf(_T("WMMEDIASUBTYPE_I420 (YV12)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_IYUV)  _tprintf(_T("WMMEDIASUBTYPE_IYUV (YV12)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_YV12)  _tprintf(_T("WMMEDIASUBTYPE_YV12 (YV12)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_YUY2)  _tprintf(_T("WMMEDIASUBTYPE_YUY2 (YUY2)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_UYVY)  _tprintf(_T("WMMEDIASUBTYPE_UYVY (UYVY)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_YVYU)  _tprintf(_T("WMMEDIASUBTYPE_YVYU (YVYU)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_YVU9)  _tprintf(_T("WMMEDIASUBTYPE_YVU9 (YVU9)\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_MP43)  _tprintf(_T("WMMEDIASUBTYPE_MP43\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_MP4S)  _tprintf(_T("WMMEDIASUBTYPE_MP4S\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_WMV1)  _tprintf(_T("WMMEDIASUBTYPE_WMV1\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_WMV2)  _tprintf(_T("WMMEDIASUBTYPE_WMV2\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_MSS1)      _tprintf(_T("WMMEDIASUBTYPE_MSS1\n"));
    // audio sub-types
    else if (mt->subtype == WMMEDIASUBTYPE_PCM)        _tprintf(_T("WMMEDIASUBTYPE_PCM\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_DRM)        _tprintf(_T("WMMEDIASUBTYPE_DRM\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_WMAudioV7)  _tprintf(_T("WMMEDIASUBTYPE_WMAudioV7\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_WMAudioV2)  _tprintf(_T("WMMEDIASUBTYPE_WMAudioV2\n"));
    else if (mt->subtype == WMMEDIASUBTYPE_ACELPnet)   _tprintf(_T("WMMEDIASUBTYPE_ACELPnet\n"));
    else                                                _tprintf(_T("Unknown\n"));

    _tprintf(_T("  bFixedSizeSamples:     %s\n"), mt->bFixedSizeSamples ? _T("TRUE") : _T("FALSE"));

    _tprintf(_T("  bTemporalCompression:  %s\n"), mt->bTemporalCompression ? _T("TRUE") : _T("FALSE"));

    _tprintf(_T("  lSampleSize:           %d\n"), mt->lSampleSize);

    _tprintf(_T("  formattype:    "));
    if      (mt->formattype == WMFORMAT_VideoInfo)     _tprintf(_T("WMFORMAT_VideoInfo\n"));
    else if (mt->formattype == WMFORMAT_WaveFormatEx)  _tprintf(_T("WMFORMAT_WaveFormatEx\n"));
    else if (mt->formattype == WMFORMAT_Script)        _tprintf(_T("WMFORMAT_Script\n"));
    else if (mt->formattype == WMSCRIPTTYPE_TwoStrings)_tprintf(_T("WMSCRIPTTYPE_TwoStrings\n"));
    else                                                _tprintf(_T("Unknown\n"));

    _tprintf(_T("  cbFormat:      %d\n"), mt->cbFormat);

    if (mt->formattype == WMFORMAT_VideoInfo) 
    {
        WMVIDEOINFOHEADER* vih = reinterpret_cast<WMVIDEOINFOHEADER*>(mt->pbFormat);
        dumpVIH(vih);
    }
}

//------------------------------------------------------------------------------
void WMProfiles::dumpVIH(WMVIDEOINFOHEADER* vih) const
{
    _tprintf(_T("WMVIDEOINFOHEADER:\n"));
    _tprintf(_T("  rcSource:         l=%d t=%d r=%d b=%d\n"), vih->rcSource.left, vih->rcSource.top, vih->rcSource.right, vih->rcSource.bottom);
    _tprintf(_T("  rcTarget:         l=%d t=%d r=%d b=%d\n"), vih->rcTarget.left, vih->rcTarget.top, vih->rcTarget.right, vih->rcTarget.bottom);
    _tprintf(_T("  dwBitRate:        %d\n"), vih->dwBitRate);
    _tprintf(_T("  dwBitErrorRate:   %d\n"), vih->dwBitErrorRate);
    _tprintf(_T("  AvgTimePerFrame:  %d\n"), vih->AvgTimePerFrame);
    _tprintf(_T("  bmiHeader:\n"));
    _tprintf(_T("    biSize          %d\n"), vih->bmiHeader.biSize);
    _tprintf(_T("    biWidth         %d\n"), vih->bmiHeader.biWidth);
    _tprintf(_T("    biHeight        %d\n"), vih->bmiHeader.biHeight);
    _tprintf(_T("    biPlanes        %d\n"), vih->bmiHeader.biPlanes);
    _tprintf(_T("    biBitCount      %d\n"), vih->bmiHeader.biBitCount);
    _tprintf(_T("    biCompression   %c%c%c%c\n"), 
        (BYTE)vih->bmiHeader.biCompression, 
        (BYTE)(vih->bmiHeader.biCompression>>8),
        (BYTE)(vih->bmiHeader.biCompression>>16),
        (BYTE)(vih->bmiHeader.biCompression>>24));
    _tprintf(_T("    biSizeImage     %d\n"), vih->bmiHeader.biSizeImage);
    _tprintf(_T("    biXPelsPerMeter %d\n"), vih->bmiHeader.biXPelsPerMeter);
    _tprintf(_T("    biYPelsPerMeter %d\n"), vih->bmiHeader.biYPelsPerMeter);
    _tprintf(_T("    biClrUsed       %d\n"), vih->bmiHeader.biClrUsed);
    _tprintf(_T("    biClrImportant  %d\n"), vih->bmiHeader.biClrImportant);
}

} // namespace camit

