//------------------------------------------------------------------------------
//
// Copyright (C) 2002 - 2003  Daniel Gehriger <gehriger@linkcad.com>
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------
#include "stdafx.h"
#include "WindowFinder.h"
#include "../res/resource.h"

namespace camit {

#define WINDOW_FINDER_MAIN_WINDOW_TITLE "Window Finder"
#define WINDOW_FINDER_APP_MUTEX_NAME    "WINDOWFINDERMUTEX"
#define WM_START_SEARCH_WINDOW           WM_USER + 100
#define WINDOW_FINDER_TOOL_TIP          "Window Finder"
#define BULLSEYE_CENTER_X_OFFSET  15
#define BULLSEYE_CENTER_Y_OFFSET  18

static WindowFinder* pThis = NULL;

//------------------------------------------------------------------------------
// Conststructor
//------------------------------------------------------------------------------
WindowFinder::WindowFinder() :
    moduleHandle_(NULL),
    mainWindow_(NULL),
    applicationMutex_(NULL),
    lastError_(0),
    searchWindowStarted_(FALSE),
    searchWindowCursor_(NULL),
    previousCursor_(NULL),
    foundWindow_(NULL),
    pen_(NULL)
{
    moduleHandle_ = GetModuleHandle(NULL);
    init();
    initResources();
}

//------------------------------------------------------------------------------
WindowFinder::~WindowFinder()
{
    uninit();
    uninitResources();
}

// -----------------------------------------------------------------------------
//  Synopsis:: 1. This routine launches the "Search Window" dialog box. 2. The
//  dialog box is a MODAL dialog box that will not return until the user clicks
//  on the "OK" or "Cancel" button.
// -----------------------------------------------------------------------------
HWND WindowFinder::startsearchWindowDialog(HWND hwndMain)
{
    pThis = this;
    int ret = DialogBox(moduleHandle_, // handle to application instance
        MAKEINTRESOURCE(IDD_DIALOG_SEARCH_WINDOW),
        hwndMain, // handle to owner window
        searchWindowDialogProc // pointer to dialog box procedure
        );
    pThis = NULL;
    return foundWindow_;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 
//  1. This function checks a hwnd to see if it is actually the
//  "Search Window" Dialog's or Main Window's own window or one of their
//  children. If so a FALSE will be returned so that these windows will not be
//  selected. 
//  2. Also, this routine checks to see if the hwnd to be checked is
//  already a currently found window. If so, a FALSE will also be returned to
//  avoid repetitions.
// -----------------------------------------------------------------------------
BOOL WindowFinder::checkWindowValidity(HWND hwndDialog, HWND hwndToCheck)
{
    HWND  hwndTemp = NULL;
    BOOL  bRet = TRUE;

    // The window must not be NULL.
    if (hwndToCheck == NULL) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

    // It must also be a valid window as far as the OS is concerned.
    if (IsWindow(hwndToCheck) == FALSE) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

    // Ensure that the window is not the current one which has already been
    // found.
    if (hwndToCheck == foundWindow_) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

    // It must also not be the main window itself.
    if (mainWindow_ && hwndToCheck == mainWindow_) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

    // It also must not be the "Search Window" dialog box itself.
    if (hwndToCheck == hwndDialog) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

    // It also must not be one of the dialog box's children...
    hwndTemp = GetParent(hwndToCheck);
    if ((hwndTemp == hwndDialog) || (mainWindow_ && hwndTemp == mainWindow_)) 
    {
        bRet = FALSE;
        goto CheckWindowValidity_0;
    }

CheckWindowValidity_0:
    return bRet;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 
//  1. This is the handler for WM_MOUSEMOVE messages sent to the
//     "Search Window" dialog proc. 
//
//  2. Note that we do not handle every WM_MOUSEMOVE
//     message sent. Instead, we check to see if "searchWindowStarted_" is TRUE.
//     This BOOL will be set to TRUE when the Window Searching Operation is actually
//     started. See the WM_COMMAND message handler in searchWindowDialogProc() for
//     more details. 
//
//  3. Because the "Search Window" dialog immediately captures the
//     mouse when the Search Operation is started, all mouse movement is monitored
//     by the "Search Window" dialog box. This is regardless of whether the mouse is
//     within or without the "Search Window" dialog. 
//
//  4. One important note is that
//     the horizontal and vertical positions of the mouse cannot be calculated from
//     "lParam". These values can be inaccurate when the mouse is outside the dialog
//     box. Instead, use the GetCursorPos() API to capture the position of the
//     mouse.
// -----------------------------------------------------------------------------
long WindowFinder::doMouseMove(HWND hwndDialog, UINT message, WPARAM wParam, LPARAM lParam)
{
    POINT screenpoint;
    HWND  hwndFoundWindow = NULL;
    long  lRet = 0;

    // Must use GetCursorPos() instead of calculating from "lParam".
    GetCursorPos(&screenpoint);

    // Determine the window that lies underneath the mouse cursor.
    hwndFoundWindow = WindowFromPoint(screenpoint);

    // Check first for validity.
    if (checkWindowValidity(hwndDialog, hwndFoundWindow)) 
    {

        // If there was a previously found window, we must instruct it to
        // refresh itself. This is done to remove any highlighting effects drawn
        // by us.
        if (foundWindow_) 
        {
            refreshWindow(foundWindow_);
        }

        // Indicate that this found window is now the current global found
        // window.
        foundWindow_ = hwndFoundWindow;

        // We now highlight the found window.
        highlightFoundWindow(hwndDialog, foundWindow_);
    }

    return lRet;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 1. Handler for WM_LBUTTONUP message sent to the "Search Window"
//  dialog box. 2. We restore the screen cursor to the previous one. 3. We stop
//  the window search operation and release the mouse capture.
// -----------------------------------------------------------------------------
long WindowFinder::doMouseUp(HWND hwndDialog, UINT message, WPARAM wParam, LPARAM lParam)
{
    long  lRet = 0;

    // If we had a previous cursor, set the screen cursor to the previous one.
    // The cursor is to stay exactly where it is currently located when the
    // left mouse button is lifted.
    if (previousCursor_) 
    {
        SetCursor(previousCursor_);
    }

    // If there was a found window, refresh it so that its highlighting is
    // erased.
    //  if (foundWindow_) {
    //    refreshWindow(foundWindow_);
    //  }

    // Set the bitmap on the Finder Tool icon to be the bitmap with the
    // bullseye bitmap.
    setFinderToolImage(hwndDialog, TRUE);

    // Very important : must release the mouse capture.
    ReleaseCapture();

    // Make the main window appear normally.
    ShowWindow(mainWindow_, SW_SHOWNORMAL);

    // Set the global search window flag to FALSE.
    searchWindowStarted_ = FALSE;

    return lRet;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 1. This routine sets the Finder Tool icon to contain an
//  appropriate bitmap. 2. If bSet is TRUE, we display the BullsEye bitmap.
//  Otherwise the empty window bitmap is displayed.
// -----------------------------------------------------------------------------
BOOL WindowFinder::setFinderToolImage(HWND hwndDialog, BOOL bSet)
{
    HBITMAP hBmpToSet = NULL;
    BOOL    bRet = TRUE;

    if (bSet) 
    {
        // Set a FILLED image.
        hBmpToSet = finderIconFilled_;
    }
    else 
    {
        // Set an EMPTY image.
        hBmpToSet = finderIconEmpty_;
    }

    SendDlgItemMessage((HWND) hwndDialog,                 // handle of dialog box
        (int)IDC_STATIC_ICON_FINDER_TOOL,  // identifier of control
        (UINT) STM_SETIMAGE,               // message to send
        (WPARAM) IMAGE_BITMAP,             // first message
        ///parameter
        (LPARAM) hBmpToSet // second message parameter
        );

    return bRet;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 1. This routine moves the mouse cursor hotspot to the exact centre
//  position of the bullseye in the finder tool static control. 2. This function,
//  when used together with DoSetFinderToolImage(), gives the illusion that the
//  bullseye image has indeed been transformed into a cursor and can be moved
//  away from the Finder Tool Static control.
// -----------------------------------------------------------------------------
BOOL WindowFinder::moveCursorPositionToBullsEye(HWND hwndDialog)
{
    BOOL  bRet = FALSE;
    HWND  hwndToolFinder = NULL;
    RECT  rect;
    POINT screenpoint;

    // Get the window handle of the Finder Tool static control.
    hwndToolFinder = GetDlgItem(hwndDialog, IDC_STATIC_ICON_FINDER_TOOL);

    if (hwndToolFinder) 
    {
        // Get the screen coordinates of the static control, add the appropriate
        // pixel offsets to the center of the bullseye and move the mouse cursor
        // to this exact position.
        GetWindowRect(hwndToolFinder, &rect);
        screenpoint.x = rect.left + BULLSEYE_CENTER_X_OFFSET;
        screenpoint.y = rect.top + BULLSEYE_CENTER_Y_OFFSET;
        SetCursorPos(screenpoint.x, screenpoint.y);
    }

    return bRet;
}

// -----------------------------------------------------------------------------
//  Synopsis:: 1. This function starts the window searching operation. 2. A very
//  important part of this function is to capture all mouse activities from now
//  onwards and direct all mouse messages to the "Search Window" dialog box
//  procedure.
// -----------------------------------------------------------------------------
long WindowFinder::searchWindow(HWND hwndDialog)
{
    long  lRet = 0;

    // Set the global "searchWindowStarted_" flag to TRUE.
    searchWindowStarted_ = TRUE;

    // Display the empty window bitmap image in the Finder Tool static control.
    setFinderToolImage(hwndDialog, FALSE);

    moveCursorPositionToBullsEye(hwndDialog);

    // Set the screen cursor to the BullsEye cursor.
    if (searchWindowCursor_) 
    {
        previousCursor_ = SetCursor(searchWindowCursor_);
    }
    else 
    {
        previousCursor_ = NULL;
    }

    // Very important : capture all mouse activities from now onwards and
    // direct all mouse messages to the "Search Window" dialog box procedure.
    SetCapture(hwndDialog);

    // Hide the main window.
    ShowWindow(mainWindow_, SW_HIDE);

    return lRet;
}

// -----------------------------------------------------------------------------
long WindowFinder::refreshWindow(HWND hwndWindowToBeRefreshed)
{
    long  lRet = 0;

    InvalidateRect(hwndWindowToBeRefreshed, NULL, TRUE);
    UpdateWindow(hwndWindowToBeRefreshed);
    RedrawWindow(hwndWindowToBeRefreshed,
        NULL,
        NULL,
        RDW_FRAME | RDW_INVALIDATE | RDW_UPDATENOW | RDW_ALLCHILDREN);

    return lRet;
}

// -----------------------------------------------------------------------------
//  Performs a highlighting of a found window. Comments below will demonstrate
//  how this is done.
// -----------------------------------------------------------------------------
long WindowFinder::highlightFoundWindow(HWND hwndDialog, HWND hwndFoundWindow)
{
    HDC     hWindowDC = NULL;   // The DC of the found window.
    HGDIOBJ hPrevPen = NULL;    // Handle of the existing pen in the DC of the
    ///found window.
    HGDIOBJ hPrevBrush = NULL;  // Handle of the existing brush in the DC of the
    ///found window.
    RECT    rect;               // Rectangle area of the found window.
    long    lRet = 0;

    // Get the screen coordinates of the rectangle of the found window.
    GetWindowRect(hwndFoundWindow, &rect);

    // Get the window DC of the found window.
    hWindowDC = GetWindowDC(hwndFoundWindow);

    if (hWindowDC) 
    {
        // Select our created pen into the DC and backup the previous pen.
        hPrevPen = SelectObject(hWindowDC, pen_);

        // Select a transparent brush into the DC and backup the previous brush.
        hPrevBrush = SelectObject(hWindowDC, GetStockObject(HOLLOW_BRUSH));

        // Draw a rectangle in the DC covering the entire window area of the
        // found window.
        Rectangle(hWindowDC, 0, 0, rect.right - rect.left, rect.bottom - rect.top);

        // Reinsert the previous pen and brush into the found window's DC.
        SelectObject(hWindowDC, hPrevPen);

        SelectObject(hWindowDC, hPrevBrush);

        // Finally release the DC.
        ReleaseDC(hwndFoundWindow, hWindowDC);
    }

    return lRet;
}

// -----------------------------------------------------------------------------
BOOL CALLBACK WindowFinder::searchWindowDialogProc(HWND    hwndDlg, // handle to dialog box
    UINT    uMsg,    // message
    WPARAM  wParam,  // first message parameter
    LPARAM  lParam // second message parameter
    )
{
    BOOL  bRet = FALSE;                     // Default return value.

    switch (uMsg)
    {
    case WM_INITDIALOG: 
        bRet = TRUE;
        break;

    case WM_MOUSEMOVE: 
        bRet = TRUE;

        if (pThis->searchWindowStarted_) 
        {
            // Only when we have started the Window Searching operation will we
            // track mouse movement.
            pThis->doMouseMove(hwndDlg, uMsg, wParam, lParam);
        }
        break;

    case WM_LBUTTONUP: 
        bRet = TRUE;

        if (pThis->searchWindowStarted_) 
        {
            // Only when we have started the window searching operation will we
            // be interested when the user lifts up the left mouse button.
            pThis->doMouseUp(hwndDlg, uMsg, wParam, lParam);
        }
        break;

    case WM_COMMAND: 
        {
            WORD  wNotifyCode = HIWORD(wParam); // notification code
            WORD  wID = LOWORD(wParam);         // item, control, or accelerator
            ///identifier
            HWND  hwndCtl = (HWND) lParam;      // handle of control

            if ((wID == IDOK) || (wID == IDCANCEL)) 
            {
                bRet = TRUE;

                if (pThis->foundWindow_) 
                {
                    pThis->refreshWindow(pThis->foundWindow_);
                }

                if (wID == IDCANCEL) 
                {
                    pThis->foundWindow_ = NULL;
                }

                EndDialog(hwndDlg, wID);
            }

            if (wID == IDC_STATIC_ICON_FINDER_TOOL) 
            {
                // Because the IDC_STATIC_ICON_FINDER_TOOL static control is set
                // with the SS_NOTIFY flag, the Search Window's dialog box will be
                // sent a WM_COMMAND message when this static control is clicked.
                bRet = TRUE;

                // We start the window search operation by calling the
                // DosearchWindow() function.
                pThis->searchWindow(hwndDlg);
            }
        }
        break;

    default:
        bRet = FALSE;
        break;
    }

    return bRet;
}

// -----------------------------------------------------------------------------
BOOL WindowFinder::init()
{
    BOOL  bRetTemp = FALSE;
    long  lRetTemp = 0;
    BOOL  bRet = FALSE;

    // Create the application mutex so that no two instances of this app can run.
    applicationMutex_ = CreateMutex((LPSECURITY_ATTRIBUTES) NULL,
        (BOOL) TRUE,
        (LPCTSTR) WINDOW_FINDER_APP_MUTEX_NAME);

    lastError_ = GetLastError();

    // If cannot create Mutex, exit.
    if (applicationMutex_ == NULL) 
    {
        bRet = FALSE;
        goto InitializeApplication_0;
    }

    // If Mutex already existed, exit.
    if (lastError_ == ERROR_ALREADY_EXISTS) 
    {
        CloseHandle(applicationMutex_);
        applicationMutex_ = NULL;
        bRet = FALSE;
        goto InitializeApplication_0;
    }

    // All went well, return a TRUE.
    bRet = TRUE;

InitializeApplication_0:
    return bRet;
}

// -----------------------------------------------------------------------------
BOOL WindowFinder::uninit()
{
    BOOL  bRet = FALSE;

    bRet = UnregisterClass(_T("WindowFinderMainWindow"), moduleHandle_);

    if (applicationMutex_) 
    {
        ReleaseMutex(applicationMutex_);
        CloseHandle(applicationMutex_);
        applicationMutex_ = NULL;
    }

    return bRet;
}

// -----------------------------------------------------------------------------
BOOL WindowFinder::initResources()
{
    BOOL  bRet = FALSE;

    searchWindowCursor_ = LoadCursor(moduleHandle_, MAKEINTRESOURCE(IDC_CURSOR_SEARCH_WINDOW));
    if (searchWindowCursor_ == NULL) 
    {
        bRet = FALSE;
        goto InitialiseResources_0;
    }

    pen_ = CreatePen(PS_SOLID, 3, RGB(256, 0, 0));
    if (pen_ == NULL) 
    {
        bRet = FALSE;
        goto InitialiseResources_0;
    }

    finderIconFilled_ = LoadBitmap(moduleHandle_, MAKEINTRESOURCE(IDB_BITMAP_FINDER_FILLED));
    if (finderIconFilled_ == NULL) 
    {
        bRet = FALSE;
        goto InitialiseResources_0;
    }

    finderIconEmpty_ = LoadBitmap(moduleHandle_,
        MAKEINTRESOURCE(IDB_BITMAP_FINDER_EMPTY));
    if (finderIconEmpty_ == NULL) 
    {
        bRet = FALSE;
        goto InitialiseResources_0;
    }

    // All went well. Return TRUE.
    bRet = TRUE;

InitialiseResources_0:
    return bRet;
}

// -----------------------------------------------------------------------------
BOOL WindowFinder::uninitResources()
{
    BOOL  bRet = TRUE;

    if (searchWindowCursor_) 
    {
        // No need to destroy searchWindowCursor_. It was not created using
        // CreateCursor().
    }

    if (pen_) 
    {
        bRet = DeleteObject(pen_);
        pen_ = NULL;
    }

    if (finderIconFilled_) 
    {
        DeleteObject(finderIconFilled_);
        finderIconFilled_ = NULL;
    }

    if (finderIconEmpty_) 
    {
        DeleteObject(finderIconEmpty_);
        finderIconEmpty_ = NULL;
    }

    return bRet;
}

} // namespace camit
